/****************************************************************************/
/* MESH.CC - Dreiecks-Meshes (Phong-Schattiert)                             */
/*     Copyright(c) 1999 Martin Melcher, Christian Volmer                   */
/* martin@raytracer.de      Gluckstr. 24/42655 Solingen/Germany             */
/* cvolmer@gmx.de                                                           */
/* This program is free software; you can redistribute it and/or modify it  */
/* under the terms of the GNU General Public License as published by the    */
/* Free Software Foundation; either version 2 of the License, or (at your   */
/* option) any later version.                                               */
/* This Program is distributed in the hope that it will be useful, but      */
/* WITHOUT ANY WARRANTY; without even the implied warranty of               */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPUOSE. See the GNU General*/
/* Public License for more details.                                         */
/* You should have received a copy of the GNU General Publiv License along  */
/* with this program; if not, write to the Free Software Foundation, Inc.,  */
/* 675 Mass Ave, Cambridge, MA 02139, USA.                                  */
/*                                                                          */
/****************************************************************************/

#include<math.h>
#include<stdlib.h>
#include<string.h>
#include"objects.h"
#include"minigauss.h"

TMesh::TMesh()
{
  vertexno=trino=lastvertex=cubetri_no=BoundingCube_no=0;
  vertex=NULL;normal=NULL;triangle=NULL;
  BoundingCube=NULL; cube_tri=NULL;
}

TMesh::~TMesh()
{
  int i;

  for (i=0;i<BoundingCube_no;++i) delete BoundingCube[i].Cube;
  free(vertex);
  free(normal);
  free(triangle);
  free(BoundingCube); free(cube_tri);
}

int TMesh::AddVertex(TVector const &p, TVector const &norm)
{
  int i;

  for (i=vertexno-1;i>=lastvertex;--i)
  {
    if ((fabs(p.x-vertex[i].x)<1e-6)&&(fabs(p.y-vertex[i].y)<1e-6)&&(fabs(p.z-vertex[i].z)<1e-6))
    {
      normal[i]+=norm;
      return i;
    }
  }
  if (((vertexno+1)%32)==1)
  {
    vertex=(TVector *)realloc(vertex, sizeof(TVector)*(vertexno+33));
    normal=(TVector *)realloc(normal, sizeof(TVector)*(vertexno+33));
  }
  vertex[vertexno]=p;
  normal[vertexno]=norm;
  if (p.x<min.x) min.x=p.x; if (p.y<min.y) min.y=p.y; if (p.z<min.z) min.z=p.z;
  if (p.x>max.x) max.x=p.x; if (p.y>max.y) max.y=p.y; if (p.z>max.z) max.z=p.z;
  return vertexno++;
}

void TMesh::AddTriangle(TVector &p1, TVector &p2, TVector &p3)
{
  int a,b,c;
  if (((trino+1)%32)==1)
    triangle=(TMeshTriangle*)realloc(triangle, sizeof(*triangle)*(trino+33));

  TVector v1, v2, n;

  v1=p2-p1;
  v2=p3-p2;
  n.init(v1,v2);
  n.normalize();

  triangle[trino].dist=n*p1;
  triangle[trino].norm=n;

//Stelle die Einheitsvektoren durch Linearkombinationen von p1, p2, p3 dar
  MiniGauss(p1, p2, p3, TVector(1, 0, 0), triangle[trino].transformed_ex);
  MiniGauss(p1, p2, p3, TVector(0, 1, 0), triangle[trino].transformed_ey);
  MiniGauss(p1, p2, p3, TVector(0, 0, 1), triangle[trino].transformed_ez);

  triangle[trino].p1=AddVertex(p1,n);
  triangle[trino].p2=AddVertex(p2,n);
  triangle[trino++].p3=AddVertex(p3,n);
}

void TMesh::NewSurface(void)
{
  lastvertex=vertexno;
}

void TMesh::NormalizeVects(void)
{
  for (int i=0; i<vertexno; ++i)
    normal[i].normalize();
}


int TMesh::AddCubeTriangle(int trino)
{
  if ((cubetri_no+1)%32==1)
    cube_tri=(int *)realloc(cube_tri, sizeof(int)*(cubetri_no+33));

  cube_tri[cubetri_no]=trino;
  return cubetri_no++;
}

char TriangleCutsCube(TCube *TheCube, TVector &v1, TVector &v2, TVector &v3)
{                  // PROVISORISCH!!!!!!!!!!!!
  char ret;
  ret=1;
  if (v1.x<TheCube->pos.x-1.5*TheCube->scalex/2.0) ret=0;
  if (v1.y<TheCube->pos.y-1.5*TheCube->scaley/2.0) ret=0;
  if (v1.z<TheCube->pos.z-1.5*TheCube->scalez/2.0) ret=0;
  if (v1.x>TheCube->pos.x+1.5*TheCube->scalex/2.0) ret=0;
  if (v1.y>TheCube->pos.y+1.5*TheCube->scaley/2.0) ret=0;
  if (v1.z>TheCube->pos.z+1.5*TheCube->scalez/2.0) ret=0;
  if (ret) return 1;
  ret=1;
  if (v2.x<TheCube->pos.x-1.5*TheCube->scalex/2.0) ret=0;
  if (v2.y<TheCube->pos.y-1.5*TheCube->scaley/2.0) ret=0;
  if (v2.z<TheCube->pos.z-1.5*TheCube->scalez/2.0) ret=0;
  if (v2.x>TheCube->pos.x+1.5*TheCube->scalex/2.0) ret=0;
  if (v2.y>TheCube->pos.y+1.5*TheCube->scaley/2.0) ret=0;
  if (v2.z>TheCube->pos.z+1.5*TheCube->scalez/2.0) ret=0;
  if (ret) return 1;
  ret=1;
  if (v3.x<TheCube->pos.x-1.5*TheCube->scalex/2.0) ret=0;
  if (v3.y<TheCube->pos.y-1.5*TheCube->scaley/2.0) ret=0;
  if (v3.z<TheCube->pos.z-1.5*TheCube->scalez/2.0) ret=0;
  if (v3.x>TheCube->pos.x+1.5*TheCube->scalex/2.0) ret=0;
  if (v3.y>TheCube->pos.y+1.5*TheCube->scaley/2.0) ret=0;
  if (v3.z>TheCube->pos.z+1.5*TheCube->scalez/2.0) ret=0;
  if (ret) return 1;

/*  if (TheCube->callisinobj(v1)) return 1;
  if (TheCube->callisinobj(v2)) return 1;
  if (TheCube->callisinobj(v3)) return 1;*/
  return 0;
}

int TMesh::AddBoundingCube(TCube *Cube, int *FoundTris, int FoundNo)
{
  int i, k;
  int MyNumber;
  int *trilist;
  int MyFoundNo;

  if ((BoundingCube_no+1)%32==1)
    BoundingCube=(TBoundingCube *)realloc(BoundingCube, sizeof(TBoundingCube)*(BoundingCube_no+33));

  BoundingCube[BoundingCube_no].Cube=Cube;

  TBoundingCube &MyCube=BoundingCube[BoundingCube_no];
  MyCube.firsttri=MyCube.lasttri=-1;

//Suchen: Welche Dreiecke liegen in diesem Wuerfel?
  if (FoundTris==NULL)
  {
    for (i=0;i<trino;++i)
    {
      if (TriangleCutsCube(Cube,vertex[triangle[i].p1],vertex[triangle[i].p2],vertex[triangle[i].p3]))
        k=AddCubeTriangle(i);
      else continue;

      MyCube.lasttri=k;
      if (MyCube.firsttri==-1) MyCube.firsttri=k;
    }
  } else
  {
    for (i=0; i<FoundNo; ++i)
    {
      if (TriangleCutsCube(Cube,vertex[triangle[FoundTris[i]].p1],vertex[triangle[FoundTris[i]].p2],vertex[triangle[FoundTris[i]].p3]))
        k=AddCubeTriangle(FoundTris[i]);
      else continue;

      MyCube.lasttri=k;
      if (MyCube.firsttri==-1) MyCube.firsttri=k;
    }
  }

  if (MyCube.firsttri==-1) //Gar keine Dreiecke drin!
  {
    delete Cube;
    return -1;
  }

  MyNumber=BoundingCube_no++;
  if ((MyCube.lasttri-MyCube.firsttri>30)          //Noch zu viele Dreiecke? -> Weiter unterteilen!
      &&((FoundTris==NULL)||(MyCube.lasttri-MyCube.firsttri<FoundNo/1.5)))
  {
    MyFoundNo=MyCube.lasttri-MyCube.firsttri+1;
    trilist=(int *)malloc(sizeof(int)*MyFoundNo); //Gefundene Dreiecke in Liste kopieren, damit die Unterwuerfel
                                                  //nicht mehr so viel zum Testen haben.
    memcpy(trilist,cube_tri+MyCube.firsttri, sizeof(int)*MyFoundNo);
    cubetri_no=MyCube.firsttri;                   //Liste mit Dreiecken wieder freigeben!
    MyCube.firsttri=-1;
    SplitCube(MyNumber, trilist, MyFoundNo);      //SplitCube kann BoundingCube_no verndern!
    free(trilist);
    return MyNumber;
  }
  else
    for (i=0;i<8;++i) MyCube.subcube[i]=-1;

  return MyNumber;
}

void TMesh::SplitCube(int CubeNo, int *FoundTris, int FoundNo)
{ //O=oben U=unten L=links R=rechts V=vorne H=hinten
  TCube *CubeOLV=new TCube(), *CubeORV=new TCube(), *CubeOLH=new TCube(), *CubeORH=new TCube();
  TCube *CubeULV=new TCube(), *CubeURV=new TCube(), *CubeULH=new TCube(), *CubeURH=new TCube();

  TCube *MyCube=BoundingCube[CubeNo].Cube;

  CubeULV->pos.x=CubeULH->pos.x=CubeOLH->pos.x=CubeOLV->pos.x=MyCube->pos.x-MyCube->scalex/4.0;
  CubeURV->pos.x=CubeURH->pos.x=CubeORH->pos.x=CubeORV->pos.x=MyCube->pos.x+MyCube->scalex/4.0;

  CubeOLV->pos.y=CubeOLH->pos.y=CubeORV->pos.y=CubeORH->pos.y=MyCube->pos.y-MyCube->scaley/4.0;
  CubeULV->pos.y=CubeULH->pos.y=CubeURV->pos.y=CubeURH->pos.y=MyCube->pos.y+MyCube->scaley/4.0;

  CubeOLV->pos.z=CubeORV->pos.z=CubeULV->pos.z=CubeURV->pos.z=MyCube->pos.z-MyCube->scalez/4.0;
  CubeOLH->pos.z=CubeORH->pos.z=CubeULH->pos.z=CubeURH->pos.z=MyCube->pos.z+MyCube->scalez/4.0;

  CubeOLV->scalex=CubeORV->scalex=CubeOLH->scalex=CubeORH->scalex=CubeULV->scalex=CubeURV->scalex=CubeULH->scalex=CubeURH->scalex=MyCube->scalex/2.0;
  CubeOLV->scaley=CubeORV->scaley=CubeOLH->scaley=CubeORH->scaley=CubeULV->scaley=CubeURV->scaley=CubeULH->scaley=CubeURH->scaley=MyCube->scaley/2.0;
  CubeOLV->scalez=CubeORV->scalez=CubeOLH->scalez=CubeORH->scalez=CubeULV->scalez=CubeURV->scalez=CubeULH->scalez=CubeURH->scalez=MyCube->scalez/2.0;

  BoundingCube[CubeNo].subcube[0]=AddBoundingCube(CubeULV, FoundTris, FoundNo);
  BoundingCube[CubeNo].subcube[1]=AddBoundingCube(CubeURV, FoundTris, FoundNo);
  BoundingCube[CubeNo].subcube[2]=AddBoundingCube(CubeULH, FoundTris, FoundNo);
  BoundingCube[CubeNo].subcube[3]=AddBoundingCube(CubeURH, FoundTris, FoundNo);
  BoundingCube[CubeNo].subcube[4]=AddBoundingCube(CubeOLV, FoundTris, FoundNo);
  BoundingCube[CubeNo].subcube[5]=AddBoundingCube(CubeORV, FoundTris, FoundNo);
  BoundingCube[CubeNo].subcube[6]=AddBoundingCube(CubeOLH, FoundTris, FoundNo);
  BoundingCube[CubeNo].subcube[7]=AddBoundingCube(CubeORH, FoundTris, FoundNo);

}

void TMesh::InitCubes(void)
{
  TCube *FirstCube=new TCube();
  FirstCube->pos.x=(max.x+min.x)/2.0;
  FirstCube->pos.y=(max.y+min.y)/2.0;
  FirstCube->pos.z=(max.z+min.z)/2.0;
  FirstCube->scalex=max.x-min.x;
  FirstCube->scaley=max.y-min.y;
  FirstCube->scalez=max.z-min.z;
  AddBoundingCube(FirstCube, NULL, 0);
}


void TMesh::Done()
{
  NormalizeVects();
  InitCubes();
}



/*****************************************************************************/

char TMesh::Obj_NextCut(double &t)
{
  TVector cut;
  rayid++;
  return cube_firstcut(0, nstart, ndir, cut, lastnorm, t, nmaxt);
}


char TMesh::cube_firstcut(int cubeno, TVector const &start, TVector const &dir, TVector &cut, TVector &norm, double &p_t, double maxt)
{
  if (BoundingCube[cubeno].firsttri!=-1)
    return triangle_firstcut(BoundingCube[cubeno].firsttri, BoundingCube[cubeno].lasttri, start, dir, cut, norm, p_t, maxt);

//Wuerfel ist offensichtlich weiter unterteilt.

//1.: Sinnvolle Reihenfolge bestimmen, in welcher die Unterwuerfel durchgegangen werden
  int cubes[8], cubes_no=0, tested=-1;
  double dist[8], curdist;
  TVector tmp;
  int i,j;
  TBoundingCube &MyCube=BoundingCube[cubeno];

  memcpy(cubes,MyCube.subcube, 8*sizeof(int));

  for (i=0;i<8;++i)
  {
    if (MyCube.subcube[i]==-1) continue; //Leerer Wuerfel

    if (BoundingCube[MyCube.subcube[i]].rayid!=rayid)
    {
      BoundingCube[MyCube.subcube[i]].rayid=rayid;
      BoundingCube[MyCube.subcube[i]].Cube->SetRay(start, dir, DBL_MAX);
    }
    if (BoundingCube[MyCube.subcube[i]].Cube->NextCut(curdist))
    {
                               //Schnittpunkt gefunden? => Einsortieren!
      for (j=0; (j<cubes_no) && (dist[j]<curdist); ++j) ;

      memmove(cubes+j+1, cubes+j, sizeof(int)*(cubes_no-j));
      cubes[j]=i;
      memmove(dist+j+1, dist+j, sizeof(double)*(cubes_no-j));
      dist[j]=curdist;
      ++cubes_no;
    }
  }

  for (i=0;i<cubes_no;++i) //Sonderfall beachten: Startpunkt liegt in einem der Wuerfel
    if (BoundingCube[cubes[i]].Cube->InObject(0.0))
    {
      if (cube_firstcut(MyCube.subcube[i], start, dir, cut, norm, p_t, maxt))
        return 1;
      else tested=i;
    }

  for (i=0;i<cubes_no;++i)
  {
    if (cubes[i]==tested) continue; //Den hatten wir schon bei der Sondefallbetrachtung
    if (cube_firstcut(MyCube.subcube[cubes[i]], start, dir, cut, norm, p_t, maxt))
      return 1;
  }
  return 0;
}

char TMesh::triangle_firstcut(int firsttri, int lasttri, TVector const &start, TVector const &dir, TVector &cut, TVector &norm, double &p_t, double maxt)
{
  maxt -= 1e-3;
  int i, thetri;
  double t, thet = maxt, d;
  TVector thecut;

  TVector transformed;
  TVector TheTransform;

  for (i = firsttri; i <= lasttri; i++)                    //Alle Dreiecke zwischen firsttri und lasttri durchgehen
  {
    TMeshTriangle &tri = triangle[cube_tri[i]];
    d = dir*tri.norm;                                     //Zuerst: Schnittpunkt mit der Dreiecksebene
    if (fabs(d) < 1e-6) continue;
    t = (tri.dist - start*tri.norm) / d;

    if (t < 1e-3) continue;
    if (t > thet) continue;

    cut=start+dir*t;

    transformed=tri.transformed_ex*cut.x+tri.transformed_ey*cut.y+tri.transformed_ez*cut.z;

    if ((transformed.x<0) || (transformed.x>1)) continue;
    if ((transformed.y<0) || (transformed.y>1)) continue;
    if ((transformed.z<0) || (transformed.z>1)) continue;

    TheTransform=transformed;
    thet = t;
    thecut = cut;
    thetri = cube_tri[i];
  }

  if (thet == maxt) return 0;                              //OK, Schnittpunkt gefunden. Jetzt den Normalenvektor interpolieren

  norm=normal[triangle[thetri].p1]*TheTransform.x
      +normal[triangle[thetri].p2]*TheTransform.y
      +normal[triangle[thetri].p3]*TheTransform.z;
  norm.normalize();
  cut=thecut;
  p_t=thet;

  return 1;
}



char TMesh::Obj_InObject(double t)      //Noch nicht implementiert.
{
  return 0;
}

TVector TMesh::Obj_GetNormVect(void)
{
  return lastnorm;
}


/*****************************************************************************/

TMesh *CreateTorus(double rad1, double rad2, int segments, int slices)
{
  int i, j;
  double x1, y1, z1, x2, y2, z2, x3, y3, z3, x4, y4, z4, x5, y5, z5;
  TVector v1,v2,v3,v4;
  TMesh *NewMesh=new TMesh();

  for (i = 0; i < slices; i++)
  {
    for (j = 0; j < segments; j++)
    {
      x1 = rad1 + rad2 * sin(j*2*M_PI/segments);
      y1 =        rad2 * cos(j*2*M_PI/segments);
      z1 = 0.0;
      x2 = rad1 + rad2 * sin((j+1)*2*M_PI/segments);
      y2 =        rad2 * cos((j+1)*2*M_PI/segments);
      z2 = 0.0;

      x3 = x1*cos(i*2*M_PI/slices);
      y3 = y1;
      z3 = x1*sin(i*2*M_PI/slices);

      x4 = x2*cos(i*2*M_PI/slices);
      y4 = y2;
      z4 = x2*sin(i*2*M_PI/slices);

      x5 = x1*cos((i+1)*2*M_PI/slices);
      z5 = x1*sin((i+1)*2*M_PI/slices);
      x1 = x5; z1 = z5;

      x5 = x2*cos((i+1)*2*M_PI/slices);
      z5 = x2*sin((i+1)*2*M_PI/slices);
      x2 = x5; z2 = z5;

      v1.init(x1, y1, z1);
      v2.init(x2, y2, z2);
      v3.init(x3, y3, z3);
      v4.init(x4, y4, z4);

      NewMesh->AddTriangle(v1, v2, v3);
      NewMesh->AddTriangle(v2, v4, v3);

      //  3         1
      //    +-----+
      //    | \   |
      //    |   \ |
      //    +-----+
      //  4         2
    }
  }
  NewMesh->Done();
  return NewMesh;
}
