/****************************************************************************/
/* OBJECTS.CPP - Objektdefinitionen                                         */
/* Alle Objekte verden mit Einheitsgre und im Ursprung berechnet,         */
/* Der Rest wird in RAY20.CPP besorgt                                       */
/*    Copyright (c) 1999 Martin Melcher                                     */
/* martin@raytracer.de      Gluckstr. 24/42655 Solingen/Germany             */
/* This program is free software; you can redistribute it and/or modify it  */
/* under the terms of the GNU General Public License as published by the    */
/* Free Software Foundation; either version 2 of the License, or (at your   */
/* option) any later version.                                               */
/* This Program is distributed in the hope that it will be useful, but      */
/* WITHOUT ANY WARRANTY; without even the implied warranty of               */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPUOSE. See the GNU General*/
/* Public License for more details.                                         */
/* You should have received a copy of the GNU General Publiv License along  */
/* with this program; if not, write to the Free Software Foundation, Inc.,  */
/* 675 Mass Ave, Cambridge, MA 02139, USA.                                  */
/*                                                                          */
/****************************************************************************/

#include"ray20.h"
#include"tvector.h"
#include<math.h>
#include"objects.h"
#include<stdlib.h>
#include<stdarg.h>

#define sqr(x) ((x)*(x))


/*********TSphere:Kugel******************************************************/

char TSphere::Obj_NextCut(double &t)
{
  //Geradengleichungen: {x,y,z}=n{x,y,z}+m{x,y,z}*t
  //Mit n{x,y,z} = start.{x,y,z} und m{x,y,z} = dir.{x,y,z}

  double a,b,c;
  double determ;

/*  a=sqr(dir.x)+sqr(dir.y)+sqr(dir.z);
  b=2*dir.x*start.x+2*dir.y*start.y+2*dir.z*start.z;
  c=sqr(start.x)+sqr(start.y)+sqr(start.z)-1;*/
  if (isfirst){
    onecut=0;
    a=ndir*ndir;
    b=(ndir*nstart)*2;
    c=nstart*nstart-1;

    determ=sqr(b)-4*a*c;
    if (determ<0) return 0;

    t1=(-b-sqrt(determ))/(2*a);
    t2=(-b+sqrt(determ))/(2*a);

    if (t2<=lastt) return 0;
  }

  t=t1; if (t<=lastt) t=t2;
  if (t<=lastt) return 0;
  if (t>nmaxt) return 0;

  return 1;
}

TVector TSphere::Obj_GetNormVect(void)
{
  return nstart+ndir*lastt; //Bei der Kugel ist Normalvektor=Schnittpunkt
}

char TSphere::Obj_InObject(double t)
{
  TVector p=nstart+ndir*t;
  return p.len2()<=1;
}

void TSphere::calcsphere()
{
  spherem.x=spherem.y=spherem.z=0;
  spherer2=1;
  spherecalculated=1;
}

/*********TIntersection : Schnittkrper beliebiger anderer Krper************/
TIntersection::TIntersection():T3dObject()
{
  numobj=0;
  Object=0;
  type=TYPE_INTERSECTION;
  needssubprocessing=1;
}

TIntersection::~TIntersection()
{
  int i;
  for (i=0;i<numobj;++i)
    delete Object[i];
  free(Object);
}

void TIntersection::addobject(PT3dObject NewObject)
{
  Object=(PT3dObject *)realloc(Object,sizeof(PT3dObject)*++numobj);
  lastobject=Object[numobj-1]=NewObject;
}

void TIntersection::SubProcessRay()
{
  for (int i=0; i<numobj; ++i) Object[i]->SetRay(nstart, ndir, nmaxt);
}

char TIntersection::Obj_NextCut(double &t)
{
  int i,j,count;
  double shortestdist=DBL_MAX,dist;

  found=-1;

  for (i=0; i<numobj; ++i) Object[i]->SetStart(lastt);

//Suche einen Schnittpunkt, der in allen anderen Objekten drin liegt!
  for (i=0;i<numobj;++i)
  {
//So lange suchen, bis einer gefunden ist.
    count=0;
inter_nextcut:
    if ((count>10)&&(Object[i]->type!=TYPE_DUST)) continue;        // Da wird wohl nichts mehr kommen....
    if (Object[i]->NextCut(dist))
    {
      if (dist>shortestdist) continue;
      for (j=0;j<numobj;++j)
      {
        if (j==i) continue;
        if (!Object[j]->InObject(dist))
        {
          ++count;
          goto inter_nextcut;
        }
      }
      if (dist<shortestdist)
      {
        shortestdist=dist;
        found=i;
      }
    }
  }
  if (found==-1) return 0;
  t=shortestdist;

  clrred=Object[found]->clrred;
  clrgreen=Object[found]->clrgreen;
  clrblue=Object[found]->clrblue;
  light=Object[found]->light;
  flat=Object[found]->flat;
  reflect=Object[found]->reflect;
  trans=Object[found]->trans;
  brech=Object[found]->brech;
  phongsize=Object[found]->phongsize;
  subtype=Object[found]->type;
  return 1;
}

char TIntersection::Obj_InObject(double t)
{
  int i;
  for (i=0;i<numobj;++i)
    if (!Object[i]->InObject(t)) return 0;

  return 1;
}

void TIntersection::calcsphere()
{
  int i;
  double a,b,c;
  for (i=0;i<numobj;++i)
  {
    Object[i]->calcsphere();
    if (!Object[i]->spherecalculated) continue;
    spherem.x=Object[i]->pos.x+Object[i]->scalex*Object[i]->spherem.x;
    spherem.y=Object[i]->pos.y+Object[i]->scaley*Object[i]->spherem.y;
    spherem.z=Object[i]->pos.z+Object[i]->scalez*Object[i]->spherem.z;

    a=sqrt(Object[i]->spherer2)*Object[i]->scalex;
    b=sqrt(Object[i]->spherer2)*Object[i]->scaley;
    c=sqrt(Object[i]->spherer2)*Object[i]->scalez;
    spherer2=a; if (b>spherer2) spherer2=b; if (c>spherer2) spherer2=c;
    spherer2=sqr(spherer2);
    spherecalculated=1;
    return;
  }
  return;
}

TVector TIntersection::Obj_GetNormVect(void)
{
  return Object[found]->GetNormVect();
}

/*********TContainer : Objekte einfach "Zusammenschmeien" ******************/

char TContainer::Obj_NextCut(double &t)
{
  int i;
  double shortestdist=DBL_MAX,dist;

  found=-1;

  for (i=0; i<numobj; ++i) Object[i]->SetStart(lastt);

  for (i=0;i<numobj;++i)
  {
    if (Object[i]->NextCut(dist))
    {
      if (dist<shortestdist)
      {
        shortestdist=dist;
        found=i;
      }
    }
  }
  if (found==-1) return 0;
  t=shortestdist;

  clrred=Object[found]->clrred;
  clrgreen=Object[found]->clrgreen;
  clrblue=Object[found]->clrblue;
  light=Object[found]->light;
  flat=Object[found]->flat;
  reflect=Object[found]->reflect;
  trans=Object[found]->trans;
  brech=Object[found]->brech;
  phongsize=Object[found]->phongsize;
  subtype=Object[found]->type;
  if (Object[found]->subtype>=0)
    subtype=Object[found]->subtype;
  return 1;
}

char TContainer::Obj_InObject(double t)
{
  int i;
  for (i=0;i<numobj;++i)
    if (Object[i]->InObject(t)) return 1;

  return 0;
}

void TContainer::calcsphere()
{
/*  double dist,a,b,c;
  int i;

  spherem.x=spherem.y=spherem.z=0;
  for (i=0;i<numobj;++i)
  {
    Object[i]->calcsphere();
    if (!Object[i]->spherecalculated) return;
    spherem.x+=Object[i]->midx+Object[i]->spherem.x*Object[i]->scalex;
    spherem.y+=Object[i]->midy+Object[i]->spherem.y*Object[i]->scaley;
    spherem.z+=Object[i]->midz+Object[i]->spherem.z*Object[i]->scalez;
  }
  spherem.x/=numobj;
  spherem.y/=numobj;
  spherem.z/=numobj;
  spherer2=0;
  for (i=0;i<numobj;++i)
  {
    a=sqrt(Object[i]->spherer2)*Object[i]->scalex;
    b=sqrt(Object[i]->spherer2)*Object[i]->scaley;
    c=sqrt(Object[i]->spherer2)*Object[i]->scalez;
    dist=a; if (b>dist) dist=b; if (c>dist) dist=c;
    dist+=sqrt(sqr(Object[i]->midx)+sqr(Object[i]->midy)+sqr(Object[i]->midz));
    if (dist>spherer2) spherer2=dist;
  }
  spherer2=sqr(spherer2);
  spherecalculated=1;*/

  nosphere=1;
}

/************* TPolygon : Polygone (auf einer Ebene) ************************/

TPolygon::TPolygon()
{
  p=NULL;
  numpoint=0;
  type=TYPE_POLYGON;
}

TPolygon::~TPolygon()
{
  free(p);
}

TPolygon::TPolygon(TVector *point1, TVector *point2, TVector *point3, ...)
{
  va_list points;
  TVector *newv;

  p=NULL;
  numpoint=0;

  addpoint(*point1);addpoint(*point2);addpoint(*point3);

  va_start(points,point3);
  while ((newv=va_arg(points,TVector *))!=NULL)
    addpoint(*newv);

  type=TYPE_POLYGON;
}

void TPolygon::addpoint(TVector &np)
{
  p=(TVector *)realloc(p,sizeof(TVector)*++numpoint);
  p[numpoint-1]=np;

  if (numpoint==3)
  {
    normvect.init(p[2]-p[0], p[1]-p[0]);
    normvect.normalize();
  }
}

char TPolygon::Obj_NextCut(double &t)
{
  double z,n,anglesum=0,angle;
  TVector v1,v2,cut;
  int i;

  nocuts=1; //Nur ein Schnittpunkt, keine weiteren

  v1=p[0]-nstart;
  n=normvect*ndir;
  if (n==0) return 0;
  z=normvect*v1;
  t=z/n;
  if (t<=lastt || t>nmaxt) return 0;

  cut=nstart+ndir*t;
  v2=p[0]-cut;
  for (i=0;i<numpoint;++i)
  {
    v1=v2;
    v2=p[(i+1)%numpoint]-cut;
    angle=acos(v1.cosangle(v2));
    if (fabs(angle-M_PI)<1e-8) return 1;
    anglesum+=angle;
  }
  if (fabs(anglesum-2.0*M_PI)>1e-8) return 0;
  return 1;
}

TVector TPolygon::Obj_GetNormVect(void)
{
  return normvect;
}

char TPolygon::Obj_InObject(double t)
{
  TVector point=nstart+ndir*t;
  TVector dir=p[0]-point;
  dir.normalize();
  if (dir*normvect>=0) return 1;
  return 0;
}

TTriangle::TTriangle(TVector &point1, TVector &point2, TVector &point3)
{
  p1=point1;
  p2=point2;
  p3=point3;
  v1=point2-point1;
  v2=point3-point2;
  v3=point1-point3;
  normvect.init(v1,v2);
  normvect/=-(double)normvect;
}

char TTriangle::Obj_NextCut(double &t)
{
  double z,n;
  TVector tmp1,tmp2,tmp3,tmp4,cut;

  nocuts=1; //Keine weiteren Schnittpunkte, hchstens einer

  tmp1=p1-nstart;
  n=normvect*ndir;
  if (n==0) return 0;
  z=normvect*tmp1;
  t=z/n;
  if (t<=lastt || t>nmaxt) return 0;

  cut=nstart+ndir*t;


//Diese Methode stammt aus comp.graphics.algorithms ("Point in triangle 3D", siehe auch eMail, "In - Q&A" - Folder)

  tmp1=cut-p1;
  tmp2.init(tmp1,v1);
  tmp3=cut-p2;
  tmp4.init(tmp3,v2);
  if (tmp2*tmp4<0) return 0;
  tmp3=cut-p3;
  tmp4.init(tmp3,v3);
  if (tmp2*tmp4<0) return 0;
  return 1;
}

TVector TTriangle::Obj_GetNormVect(void)
{
  return normvect;
}

char TTriangle::Obj_InObject(double t)
{
  TVector dir,point=nstart+ndir*t;
  dir=p1-point;
  dir.normalize();
  if (dir*normvect>=0) return 1;
  return 0;
}


/************* TUnion : Vereinigungskrper **********************************/

char TUnion::Obj_NextCut(double &t)
{
  int i,j,count;
  double shortest=DBL_MAX,dist;

  found=-1;

  for (i=0; i<numobj; ++i) Object[i]->SetStart(lastt);

//Es drfen nur die aeusseren Schnittpunkte geliefert werden!
  for (i=0;i<numobj;++i)
  {
    count=0;
Cont_nextcut:
    if (count>10) continue;                // Da wird wohl nichts mehr kommen!
    if (Object[i]->NextCut(dist))
    {
      if (dist>shortest) continue;
      for (j=0;j<numobj;++j)
      {
        if (i==j) continue;
        if (Object[j]->InObject(dist))
        {
          ++count;
          goto Cont_nextcut;
        }
      }
      if (dist<shortest)
      {
        shortest=dist;
        found=i;
      }
    }
  }

  if (found==-1) return 0;

  clrred=Object[found]->clrred;
  clrgreen=Object[found]->clrgreen;
  clrblue=Object[found]->clrblue;
  light=Object[found]->light;
  flat=Object[found]->flat;
  reflect=Object[found]->reflect;
  trans=Object[found]->trans;
  brech=Object[found]->brech;
  phongsize=Object[found]->phongsize;
  subtype=Object[found]->type;
  t=shortest;
  return 1;
}

char TUnion::Obj_InObject(double t)
{
  int i;
  for (i=0;i<numobj;++i)
    if (Object[i]->InObject(t)) return 1;

  return 0;
}

void TUnion::calcsphere()
{
/*  double dist,a,b,c;
  int i;

  spherem.x=spherem.y=spherem.z=0;
  for (i=0;i<numobj;++i)
  {
    Object[i]->calcsphere();
    if (!Object[i]->spherecalculated) return;
    spherem.x+=Object[i]->midx+Object[i]->spherem.x*Object[i]->scalex;
    spherem.y+=Object[i]->midy+Object[i]->spherem.y*Object[i]->scaley;
    spherem.z+=Object[i]->midz+Object[i]->spherem.z*Object[i]->scalez;
  }
  spherem.x/=numobj;
  spherem.y/=numobj;
  spherem.z/=numobj;
  spherer2=0;
  for (i=0;i<numobj;++i)
  {
    a=sqrt(Object[i]->spherer2)*Object[i]->scalex;
    b=sqrt(Object[i]->spherer2)*Object[i]->scaley;
    c=sqrt(Object[i]->spherer2)*Object[i]->scalez;
    dist=a; if (b>dist) dist=b; if (c>dist) dist=c;
    dist+=sqrt(sqr(Object[i]->midx)+sqr(Object[i]->midy)+sqr(Object[i]->midz));
    if (dist>spherer2) spherer2=dist;
  }
  spherer2=sqr(spherer2);
  spherecalculated=1;*/
  nosphere=1;
}
/************* TDisk : Scheibe (mit Loch) ***********************************/

char TDisk::Obj_NextCut(double &t)
{
  double l;
  TVector cut;

  nocuts=1; //Nur ein Schnittpunt
  if (ndir.y==0) return 0;
  t=-nstart.y/ndir.y;
  if (t>nmaxt || t<=lastt) return 0;
  cut=nstart+ndir*t;
  if ((l=cut.len2())>1.0) return 0;
  if (l<innerr2) return 0;
  return 1;
}

TVector TDisk::Obj_GetNormVect(void)
{
  return TVector(0,-1,0);
}

char TDisk::Obj_InObject(double t)
{
  TVector v=nstart+ndir*t;
  if (v.y>=0) return 1;
  return 0;
}

void TDisk::calcsphere()
{
  spherecalculated=1;
  spherem.x=spherem.y=spherem.z=0;
  spherer2=1;
}

/************* TCylinder : Zylinder (nur Auenwand) *************************/

char TCylinder::Obj_NextCut(double &t)
{
  double determ,a,nenn;
  char issecond=0;

  if (isfirst)
  {
    a=-2.0*nstart.x*ndir.x-2.0*nstart.z*ndir.z;
    nenn=sqr(ndir.x)+sqr(ndir.z);
    if (nenn<1e-8) return 0;

    determ=2*nstart.x*ndir.x*nstart.z*ndir.z-sqr(ndir.x)*sqr(nstart.z)+sqr(ndir.x)-sqr(ndir.z)*sqr(nstart.x)+sqr(ndir.z);
    if (determ<0) return 0;

    determ=sqrt(determ);

    t1=0.5*(a-2.0*determ)/nenn;
    t2=0.5*(a+2.0*determ)/nenn;

    if (t2<=lastt) return 0;
  }

  t=t1;
  if (t<=lastt) t=t2;
  if (t<=lastt) return 0;

  if (t>nmaxt) return 0; //Damit das Objekt bei Schattenberechnung nicht
                              //Schatten "auf sich selbst" wirft, 1e-8!

Cyfc_retry:
  cut=nstart+ndir*t;

  if (cut.y<-1 || cut.y>1)
  {
    if (issecond) return 0;
    t=t2; if (t<=lastt || t>nmaxt) return 0;
    issecond=1;
    goto Cyfc_retry;
  }

  return 1;
}

TVector TCylinder::Obj_GetNormVect(void)
{
  TVector norm=cut;
  norm.y=0; norm.normalize();
  return norm;
}

char TCylinder::Obj_InObject(double t)
{
  TVector v,p=nstart+ndir*t;
  if (p.y<-1 || p.y>1) return 0;
  v=p;v.y=0;
  if (v.len2()>1) return 0;
  return 1;
}

void TCylinder::calcsphere()
{
  spherecalculated=1;
  spherem.x=spherem.y=spherem.z=0;
  spherer2=2;
}

/************* TSimpleQuad : Einfaches Quadrat ******************************/

TSimpleQuad::TSimpleQuad(int e)
{
  ebene=e;
  type=TYPE_SIMPLEQUAD;
  switch(ebene)
  {
    case ebene_xyv:
      normvect.init(0,0,-1);
      break;
    case ebene_xyh:
      normvect.init(0,0,1);
      break;
    case ebene_xzo:
      normvect.init(0,-1,0);
      break;
    case ebene_xzu:
      normvect.init(0,1,0);
      break;
    case ebene_yzl:
      normvect.init(-1,0,0);
      break;
    case ebene_yzr:
      normvect.init(1,0,0);
      break;
  }
}

char TSimpleQuad::Obj_NextCut(double &t)
{
  TVector cut;
  switch(ebene)
  {
    case ebene_xyv:
    case ebene_xyh:
      if (ndir.z==0) return 0;
      t=-nstart.z/ndir.z;
      break;
    case ebene_xzo:
    case ebene_xzu:
      if (ndir.y==0) return 0;
      t=-nstart.y/ndir.y;
      break;
    case ebene_yzl:
    case ebene_yzr:
      if (ndir.x==0) return 0;
      t=-nstart.x/ndir.x;
      break;
  }
  if (t>nmaxt || t<=lastt) return 0;
  cut=nstart+ndir*t;
  switch(ebene)
  {
    case ebene_xyv:
    case ebene_xyh:
      if (fabs(cut.x)>0.5 || fabs(cut.y)>0.5) return 0;
      break;
    case ebene_xzo:
    case ebene_xzu:
      if (fabs(cut.x)>0.5 || fabs(cut.z)>0.5) return 0;
      break;
    case ebene_yzl:
    case ebene_yzr:
      if (fabs(cut.z)>0.5 || fabs(cut.y)>0.5) return 0;
      break;
  }
  return 1;
}

TVector TSimpleQuad::Obj_GetNormVect(void)
{
  return normvect;
}

char TSimpleQuad::Obj_InObject(double t)
{
  TVector p=nstart+ndir*t;
  switch(ebene)
  {
    case ebene_xyv:
      if (p.z>=0) return 1;return 0;
    case ebene_xyh:
      if (p.z<=0) return 1;return 0;
    case ebene_xzo:
      if (p.y>=0) return 1;return 0;
    case ebene_xzu:
      if (p.y<=0) return 1;return 0;
    case ebene_yzl:
      if (p.x>=0) return 1;return 0;
    case ebene_yzr:
      if (p.x<=0) return 1;return 0;
  }
  return 123; //Damit der Compiler nicht meckert!
}

void TSimpleQuad::calcsphere()
{
/*  spherecalculated=1;
  spherem.x=spherem.y=spherem.z=0;
  spherer2=0.5;*/
}

/************* TCube : Wrfel (aus TSimpleQuads) ****************************/

TCube::TCube()
{
  addobject(new TSimpleQuad(ebene_xyv));
  Object[0]->pos.z=-0.5;
  addobject(new TSimpleQuad(ebene_xyh));
  Object[1]->pos.z=0.5;

  addobject(new TSimpleQuad(ebene_xzo));
  Object[2]->pos.y=-0.5;
  addobject(new TSimpleQuad(ebene_xzu));
  Object[3]->pos.y=0.5;

  addobject(new TSimpleQuad(ebene_yzl));
  Object[4]->pos.x=-0.5;
  addobject(new TSimpleQuad(ebene_yzr));
  Object[5]->pos.x=0.5;
  type=TYPE_CUBE;
}

char TCube::Obj_NextCut(double &t)
{
  int i;
  double shortestdist=DBL_MAX,dist;

  found=-1;

  for (i=0; i<numobj; ++i) Object[i]->SetStart(lastt);

  for (i=0;i<numobj;++i)
  {
    if (Object[i]->NextCut(dist))
    {
      if (dist<shortestdist)
      {
        shortestdist=dist;
        found=i;
      }
    }
  }
  if (found==-1) return 0;
  t=shortestdist;

  return 1;
}

char TCube::Obj_InObject(double t)
{
  TVector p=nstart+ndir*t;
  if (fabs(p.x)<=0.5 && fabs(p.y)<=0.5 && fabs(p.z)<=0.5) return 1;
  return 0;
}


void TCube::calcsphere()
{
  spherecalculated=1;
  spherem.x=spherem.y=spherem.z=0;
  spherer2=0.75;
  for (int i=0;i<numobj;++i)
  {
    Object[i]->nosphere=1;
    Object[i]->spherecalculated=0;
  }
}

/************* TTube : Rohr (2 Zylinder & 2 Scheiben) ***********************/

TTube::TTube(double radi)
{
  nullradi=0;
  if (radi==0) nullradi=1;
  else{
    addobject(new TCylinder);
    Object[0]->scalex=Object[0]->scalez=radi;
  }
  addobject(new TCylinder);

  addobject(new TDisk(radi));
  Object[2-nullradi]->pos.y=-1;
  addobject(new TDisk(radi));
  Object[3-nullradi]->pos.y=1;

  type=TYPE_TUBE;
}

char TTube::Obj_NextCut(double &t)
{
  int i;
  double shortestdist=DBL_MAX,dist;

  found=-1;

  for (i=0; i<numobj; ++i) Object[i]->SetStart(lastt);

  for (i=0;i<numobj;++i)
  {
    if (Object[i]->NextCut(dist))
    {
      if (dist<shortestdist)
      {
        shortestdist=dist;
        found=i;
      }
    }
  }
  if (found==-1) return 0;
  t=shortestdist;
  return 1;
}

TVector TTube::Obj_GetNormVect(void)
{
  TVector norm=Object[found]->GetNormVect();
  if (!nullradi)
  {
    if (found==3||found==0)            /* Innerer Zylinder und untere Scheibe */
      norm=-norm;
  } else
  {
    if (found==2)                      /* Untere Scheibe, wenn es keinen inneren Zylinder gibt */
      norm=-norm;
  }
  return norm;
}

char TTube::Obj_InObject(double t)
{
  if (!nullradi)
  {
    if (Object[1]->InObject(t) && !Object[0]->InObject(t)) return 1;
  }else{
    if (Object[0]->InObject(t)) return 1;
  }

  return 0;
}

void TTube::calcsphere()
{
  spherecalculated=1;
  spherem.x=spherem.y=spherem.z=0;
  spherer2=2;
}

/************* TSubtraction : Differenzkrper ("Herausschneiden") ***********/

TSubtraction::~TSubtraction()
{
  delete Obja;
  delete Objb;
}

void TSubtraction::SubProcessRay()
{
  Obja->SetRay(nstart, ndir, nmaxt);
  Objb->SetRay(nstart, ndir, nmaxt);
}

char TSubtraction::Obj_NextCut(double &t)
{
  double dist1=DBL_MAX,dist2=DBL_MAX,dist;
  found=0;
  char objacuts=0;

  Obja->SetStart(lastt);
//1: Finde so lange Schnittpunkte mit Obja, bis einer nicht in Objb liegt
  while (Obja->NextCut(dist))
  {
    objacuts=1;
    if (!Objb->InObject(dist))
    {
      dist1=dist;
      found=1;
      break;
    }
  }
  if (!objacuts && (nmaxt==DBL_MAX)) return 0;

//2: Finde so lange Schnittpunkte mit Objb, bis einer in Obja liegt
  Objb->SetStart(lastt);
  while (Objb->NextCut(dist))
  {
    if (found) if (dist>dist1) break;
    if (Obja->InObject(dist))
    {
      dist2=dist;
      found=1;
      break;
    }
  }

  if (!found) return 0;
  if (dist1<dist2)
  {
    clrred=Obja->clrred;
    clrgreen=Obja->clrgreen;
    clrblue=Obja->clrblue;
    light=Obja->light;
    flat=Obja->flat;
    reflect=Obja->reflect;
    trans=Obja->trans;
    brech=Obja->brech;
    phongsize=Obja->phongsize;
    subtype=Obja->type;
    found=1;
    t=dist1;
    return 1;
  } else {
    clrred=Objb->clrred;
    clrgreen=Objb->clrgreen;
    clrblue=Objb->clrblue;
    light=Objb->light;
    flat=Objb->flat;
    reflect=Objb->reflect;
    trans=Objb->trans;
    brech=Objb->brech;
    phongsize=Objb->phongsize;
    subtype=Objb->type;
    found=2;
    t=dist2;
    return 1;
  }
}

TVector TSubtraction::Obj_GetNormVect(void)
{
  return found==1?Obja->GetNormVect():-Objb->GetNormVect();
}

char TSubtraction::Obj_InObject(double t)
{
  if (Obja->InObject(t) && !Objb->InObject(t)) return 1;
  return 0;
}

void TSubtraction::calcsphere()
{
/*  double a,b,c;
  Obja->calcsphere();
  spherecalculated=Obja->spherecalculated;
  if (!spherecalculated) return;
  spherem.x=Obja->midx+Obja->spherem.x*Obja->scalex;
  spherem.y=Obja->midy+Obja->spherem.y*Obja->scaley;
  spherem.z=Obja->midz+Obja->spherem.z*Obja->scalez;
  a=sqrt(Obja->spherer2)*Obja->scalex;
  b=sqrt(Obja->spherer2)*Obja->scaley;
  c=sqrt(Obja->spherer2)*Obja->scalez;
  spherer2=a;
  if (b>spherer2) spherer2=b;
  if (c>spherer2) spherer2=c;
  spherer2=sqr(spherer2);
  Obja->spherecalculated=0;
  Obja->nosphere=1;*/
}

/************* TCone : Kegel ************************************************/

TCone::TCone(double px1, double py1, double px2, double py2)
{
  //Grenzgeradengleichung ausrechnen, aber als x=f(y)!!!
  m=(px2-px1)/(py2-py1);
  n=px1-m*py1;
  if (py1<py2)     //y1 und y2 werden nurnoch gebraucht fr die berprfung
  {                //ob der SP noch auf dem Kegel liegt.
    y1=py1;
    y2=py2;
  }
  else
  {
    y2=py1;
    y1=py2;
  }
  type=TYPE_CONE;
}

char TCone::Obj_NextCut(double &t)
{
  double determ;

  double m2,n2,dirx2,diry2,dirz2,startx2,starty2,startz2;
  double a,b;

  char issecond=0;

  if (isfirst)
  {
    m2=m*m;
    n2=n*n;
    dirx2=ndir.x*ndir.x;
    diry2=ndir.y*ndir.y;
    dirz2=ndir.z*ndir.z;
    startx2=nstart.x*nstart.x;
    starty2=nstart.y*nstart.y;
    startz2=nstart.z*nstart.z;

    determ=dirx2*m2*starty2+dirz2*m2*starty2+m2*diry2*startx2+m2*diry2*startz2-dirx2*startz2+dirx2*n2
          -dirz2*startx2+dirz2*n2+2*dirx2*m*nstart.y*n+2*nstart.x*ndir.x*nstart.z*ndir.z
          -2*nstart.x*ndir.x*m2*nstart.y*ndir.y-2*nstart.x*ndir.x*m*ndir.y*n-2*nstart.z*ndir.z*m2*nstart.y*ndir.y
          -2*nstart.z*ndir.z*m*ndir.y*n+2*dirz2*m*nstart.y*n;

    if (determ<0) return 0;

    a=dirx2+dirz2-m2*diry2;
    if (fabs(a)<1e-20) return 0;

    b=-2*nstart.x*ndir.x-2*nstart.z*ndir.z+2*m2*nstart.y*ndir.y+2*m*ndir.y*n;
    determ=sqrt(determ);
    t1=0.5*(b-2*determ)/a;
    t2=0.5*(b+2*determ)/a;
    if (t2<=lastt) return 0;
  }

  t=t1;
  if (t<=lastt) t=t2;
  if (t<=lastt) return 0;

  if (t>nmaxt) return 0;

Cofc_retry:
  cut=nstart+ndir*t;

  if (cut.y<y1 || cut.y>y2)
  {
    if (issecond) return 0;
    t=t2; if (t<=lastt || t>nmaxt) return 0;
    issecond=1;
    goto Cofc_retry;
  }

  return 1;
}

TVector TCone::Obj_GetNormVect(void)
{
  TVector norm=cut;
  norm.y=0;
  if (norm.x==0&&norm.z==0) norm.x=1;

  norm.normalize();
  norm.y=-m;
  norm.normalize();
  return norm;
}

char TCone::Obj_InObject(double t)
{
  TVector p=nstart+ndir*t;
  double a;
  if (p.y<y1||p.y>y2) return 0;
  a=m*p.y+n;
  if (sqr(p.x)+sqr(p.z)>sqr(a)) return 0;
  return 1;
}

/************* TRotObj : Rotationskrper (mehrere Kegel) ********************/

void TRotobj::SubProcessRay()
{
  for (int i=0; i<numpoint-1; ++i) cone[i]->SetRay(nstart, ndir, nmaxt);
}

TRotobj::~TRotobj()
{
  int i;
  for (i=0;i<numpoint-1;++i)
    delete cone[i];

  free(cone);
  free(swapped);
}

void TRotobj::add(double x, double y)
{
  ++numpoint;
  double x1,y1,x2,y2;
  if (numpoint>1)
  {
    swapped=(char *)realloc(swapped,numpoint-1);
    cone=(TCone **)realloc(cone,sizeof(TCone *)*(numpoint-1));
    x1=lastx;x2=x;
    y1=lasty;y2=y;
/*    if (y2<y1)
    {
      tmp=y2;y2=y1;y1=tmp;
      tmp=x2;x2=x1;x1=tmp;
      swapped[numpoint-2]=1;
    }
    else swapped[numpoint-2]=0;*/
    cone[numpoint-2]=new TCone(x1,y1,x2,y2);
    lastx=x2;
    lasty=y2;
  }
  else
  {
    lastx=x;
    lasty=y;
  }
}

char TRotobj::Obj_NextCut(double &t)
{
  int i;
  double shortest=DBL_MAX,dist;

  found=-1;

  for (i=0;i<numpoint-1;++i)
  {
    cone[i]->SetStart(lastt);
    if (cone[i]->NextCut(dist))
    {
      if (dist<shortest)
      {
        shortest=dist;
        found=i;
      }
    }
  }

  if (found==-1) return 0;
  t=shortest;
  return 1;
}

TVector TRotobj::Obj_GetNormVect(void)
{
  return cone[found]->GetNormVect();
}

char TRotobj::Obj_InObject(double t)
{
  int i;
  char found1=0;

  for (i=0;i<numpoint-1;++i)
  {
    if (cone[i]->InObject(t)) return 1;
/*    erg=cone[i]->isinobject(p); //Kein CALL... ntig!
    if (erg && swapped[i]) return 0;
    if (erg && !swapped[i]) found1=1;*/
  }
//  if (!found1) return 0;
  return 0;
}
