/****************************************************************************/
/* Rechnen mit Polynomen aus R[x]  (double - Genauigkeit)                   */
/*  Copyright   (c) 1999 Martin Melcher                                     */
/* martin@raytracer.de      Gluckstr. 24/42655 Solingen/Germany             */
/* This program is free software; you can redistribute it and/or modify it  */
/* under the terms of the GNU General Public License as published by the    */
/* Free Software Foundation; either version 2 of the License, or (at your   */
/* option) any later version.                                               */
/* This Program is distributed in the hope that it will be useful, but      */
/* WITHOUT ANY WARRANTY; without even the implied warranty of               */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPUOSE. See the GNU General*/
/* Public License for more details.                                         */
/* You should have received a copy of the GNU General Publiv License along  */
/* with this program; if not, write to the Free Software Foundation, Inc.,  */
/* 675 Mass Ave, Cambridge, MA 02139, USA.                                  */
/*                                                                          */
/****************************************************************************/

#include<string.h>
#include<math.h>
#include<stdarg.h>
#include"polynome.h"

#define FAST_NULL 1e-20

#define max(a,b) ((a)>(b)?(a):(b))

TPolynom::TPolynom()
{
  degree=0;
  for (int i=0;i<=MAXDEG;++i)
    koeff[i]=0;
}

TPolynom::TPolynom(int deg, ...)
{
  int i;
  degree=deg;
  va_list args;

  va_start(args,deg);
  for (i=0;i<=degree;++i)
    koeff[i]=va_arg(args,double);
  for (i=degree+1;i<=MAXDEG;++i)
    koeff[i]=0;

  va_end(args);
}

void TPolynom::MinimizeDeg()
{
  for (int i=degree;i>=0;--i)
    if (fabs(koeff[i])>=FAST_NULL)
    {
      degree=i;
      return;
    }
  degree=0;
}

TPolynom TPolynom::operator += (TPolynom const &A)
{
  for (int i=0;i<= (degree=max(degree,A.degree));++i)
    koeff[i]+=A.koeff[i];
  MinimizeDeg();
  return *this;
}

TPolynom TPolynom::operator -= (TPolynom const &A)
{
  for (int i=0;i<= (degree=max(degree,A.degree));++i)
    koeff[i]-=A.koeff[i];
  MinimizeDeg();
  return *this;
}

TPolynom TPolynom::operator >>= (int A)
{
  if (A<0) return (*this<<=-A);
  if (A>degree)
  {
    memset(koeff,0,sizeof(koeff[0])*(degree+1));
    degree=0;
  }
  memmove(koeff,koeff+A,sizeof(koeff[0])*((degree-=A)+1));
  memset(koeff+degree+1,0,sizeof(koeff[0])*A);
  return *this;
}

TPolynom TPolynom::operator <<= (int A)
{
  if (A<0) return (*this>>=-A);
  memmove(koeff+A,koeff,sizeof(koeff[0])*(degree+1));
  degree+=A;
  memset(koeff,0,sizeof(koeff[0])*A);
  return *this;
}

TPolynom TPolynom::operator *= (TPolynom const &A)
{
  TPolynom Tmp1,Tmp2;
  double oldkoeff[MAXDEG+1];
  int i,olddeg;

  memcpy(oldkoeff,koeff,sizeof(koeff[0])*(MAXDEG+1));
  memset(koeff,0,sizeof(koeff[0])*(degree+1));
  olddeg=degree;

  degree=0;
  Tmp1=A;
  for (i=0;i<=olddeg;++i)
  {
    Tmp2=Tmp1;
    Tmp2*=oldkoeff[i];
    *this+=Tmp2;
    if (i<olddeg) Tmp1<<=1;
  }

  return *this;
}

TPolynom TPolynom::operator *= (double A)
{
  for (int i=0;i<=degree;++i) koeff[i]*=A;
  return *this;
}

TPolynom TPolynom::operator /= (double A)
{
  for (int i=0;i<=degree;++i) koeff[i]/=A;
  return *this;
}

TPolynom TPolynom::operator % (TPolynom const &B) const
{
  TPolynom Rest,OldRest;
  OldRest=*this;

  while(OldRest.degree>=B.degree && OldRest!=0)
  {
    Rest=B;
    Rest*=OldRest.koeff[OldRest.degree]/B.koeff[B.degree];
    Rest<<=OldRest.degree-B.degree;
    OldRest-=Rest;
  }
  return OldRest;
}

TPolynom TPolynom::operator / (TPolynom const &B) const
{
  TPolynom Rest,OldRest,Erg;
  OldRest=*this;
  double d;

  Erg.degree=degree-B.degree;
  if (Erg.degree<0)
  {
    Erg.degree=0;
    return Erg;
  }

  while(OldRest.degree>=B.degree)
  {
    Rest=B;
    d=OldRest.koeff[OldRest.degree]/B.koeff[B.degree];
    Erg.koeff[OldRest.degree-B.degree]=d;
    Rest*=d;
    Rest<<=OldRest.degree-B.degree;
    OldRest-=Rest;
  }
  Erg.MinimizeDeg();
  return Erg;
}

TPolynom TPolynom::operator -()const
{
  TPolynom A;
  A=*this;
  for (int i=0;i<=degree;++i)
    A.koeff[i]=-A.koeff[i];
  return A;
}

int TPolynom::operator == (double A) const
{
  if (degree>0) return 0;
  if (fabs(koeff[0]-A)<FAST_NULL) return 1;
  return 0;
}

int TPolynom::operator != (double A) const
{
  if (degree>0) return 1;
  if (fabs(koeff[0]-A)<FAST_NULL) return 0;
  return 1;
}

TPolynom TPolynom::Diff()
{
  TPolynom B;
  B.degree=degree-1;
  for (int i=0;i<degree;++i)
    B.koeff[i]=koeff[i+1]*(i+1);
  return B;
}

TPolynom TPolynom::ggT(TPolynom const &A) const
{
  TPolynom Rest1=*this,Rest2=A;
  TPolynom Temp;
  while (Rest2!=0)
  {
    Temp=Rest2;
    Rest2=Rest1%Rest2;
    Rest1=Temp;
  }
  return Rest1;
}

double TPolynom::eval(double x) const
{
  int i;
  double erg=koeff[0];
  double pot=1;

  for (i=1;i<=degree;++i)
  {
    pot*=x;
    erg+=pot*koeff[i];
  }
  return erg;
}


