/****************************************************************************/
/* RAY2.0 - Script - Programmiersprache                                     */
/*  Copyright   (c) 1999 Martin Melcher                                     */
/* martin@raytracer.de      Gluckstr. 24/42655 Solingen/Germany             */
/* This program is free software; you can redistribute it and/or modify it  */
/* under the terms of the GNU General Public License as published by the    */
/* Free Software Foundation; either version 2 of the License, or (at your   */
/* option) any later version.                                               */
/* This Program is distributed in the hope that it will be useful, but      */
/* WITHOUT ANY WARRANTY; without even the implied warranty of               */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPUOSE. See the GNU General*/
/* Public License for more details.                                         */
/* You should have received a copy of the GNU General Publiv License along  */
/* with this program; if not, write to the Free Software Foundation, Inc.,  */
/* 675 Mass Ave, Cambridge, MA 02139, USA.                                  */
/****************************************************************************/

#include"ray20.h"
#include"objects.h"
#include"lights.h"
#include"cameras.h"
#include"textures.h"
#include"bumpmaps.h"
#include<ctype.h>
#include<string.h>
#include<stdio.h>

#ifdef IS_LINUX
#define stricmp strcasecmp
#endif

TRaytracer *TheTracer;

void normalize(char *src, char *dst)
{
  int sp,dp=0;
  for (sp=0;sp<strlen(src);++sp)
  {
    if (src[sp]!=' ' && !isspace(src[sp]))
      dst[dp++]=toupper(src[sp]);
  }
  dst[dp]=0;
}

char testcommand(char *s, char *command)
{
  char c,e;
  c=s[strlen(command)];
  s[strlen(command)]=0;
  e=stricmp(s,command);
  s[strlen(command)]=c;
  return !e;
}

int errorline=0;
char errormsg[256];

void syntaxerr(int line, char *text)
{
  errorline=line;
  strcpy(errormsg,text);
}

void extractfilename(char *src, char *dest)
{
  char tmp[100];
  strcpy(tmp,src);
  for(int i=strlen(tmp)-1;i>=0 && tmp[i]<=' ';++i) tmp[i]=0;
  strcpy(dest, strrchr(tmp,' ')+1);  
}

#pragma warn -def

#define error(t) {syntaxerr(line,t);if (fileopened) fclose(datei);return 0;}

#define COMPMAXREKUR 30

int resx=640,resy=480;

/******************************************************************************
Debug-Levels:
  0 - Keine Meldungen
  1 - Meldungen bei Berechnungen (fraktale Landschaften, Blobs, ..)
  2 - Meldungen beim Hinzufgen von Objekten, ...
  3 - Ganz viele Meldungen
******************************************************************************/



//0=Fehler
int compilescript(char *filename, void MessageFunc(int, char*), int debuglevel)
{
  int line=0;
  PT3dObject Container[COMPMAXREKUR];
  PT3dObject newobject=NULL;
  TObjectData curdata;
  int containernum=0;
  FILE *datei;
  char str[256],command[256],str3[256];
  char messagestr[256];
  char *str2;
  double a,b,c,d,e,f,g,h;
  TVector p,p2,p3;
  int i,n,cr,cg,cb,j,k,l;
  char colordef=0;
  char cameradef=0,lightdef=0;
  char ismaterial=0;
  char fileopened=0;
  char istexture=0;

  TheTracer=new TRaytracer;
  if (TheTracer==NULL) error ("Nicht gengend freier Speicher");

  curdata.light=0;curdata.flat=1;curdata.trans=0;curdata.reflect=0;curdata.brech=1;
  curdata.phongsize=0.5;

  datei=fopen(filename,"rt");
  if (datei==NULL)
    error("Eingabedatei nicht gefunden!");

  fileopened=1;
  if (debuglevel>=3) MessageFunc(0, "Scriptdatei geffnet");
  while (!feof(datei))
  {
    ++line;
    if (fgets(str,255,datei)==0) break;
    normalize(str,command);

    if (testcommand(command,"*")); //Kommentarzeile
    else if (testcommand(command,"{")) //Kommentarblock
    {
      i=1;
      while ((i>0) && !feof(datei))
      {
        ++line;
        fgets(str,255,datei);normalize(str,str3);
        if (testcommand(str3,"}")) --i;
        else if (testcommand(str3,"{")) ++i;
      }
    }

/*------------------------------Neue Objekte--------------------------------*/

    else if (testcommand(command,"NEW"))
    {
      colordef=0;
      str2=command+3;
      if (testcommand(str2,"SPHERE"))
        newobject=new TSphere;
      else if (testcommand(str2,"CYLINDER"))
        newobject=new TCylinder;

      else if (testcommand(str2,"X"))
      {
/*        TMesh *X=new TMesh();
         p.init(1, -1, 1);
        p2.init(2,  1, 1);
        p3.init(0,  1, 1);
        X->AddTriangle(p, p2, p3);
         p.init(-1, -1, 1);
        p2.init(0,  1, 1);
        p3.init(-2, 1, 1);
        X->AddTriangle(p, p2, p3);
        X->Done();

        newobject=X;*/
        newobject=CreateTorus(1.5, 0.3, 50, 50);
      }


      else if (testcommand(str2,"TUBE"))
      {
        if (sscanf(command,"NEWTUBEINNER=%lg",&a)!=1) error("Erwartet: new Tube Inner=<d>         <d>=0..1");
        if (a>1 || a<0) error("Erwartet: new Tube Inner=<d>         <d>=0..1");
        newobject=new TTube(a);
      }
      else if (testcommand(str2,"DISK"))
      {
        if (sscanf(command,"NEWDISKINNER=%lg",&a)!=1) error("Erwartet: new Disk Inner=<d>         <d>=0..1");
        if (a>1 || a<0) error("Erwartet: new Disk Inner=<d>         <d>=0..1");
        newobject=new TDisk(a);
      }
      else if (testcommand(str2,"SQELLIPSOID"))
      {
        if (sscanf(command,"NEWSQELLIPSOIDDEG=%d",&i)!=1) error ("Erwartet: new SQEllipsoid deg=<n>");
        if (i<2 || i%2==1) error("Erwartet: new SQEllipsoid deg=<n>      <n>=2,4,6,8,...");
        newobject=new TSQEllipsoid(i);
      }
      else if (testcommand(str2,"TORUS"))
      {
        if (sscanf(command,"NEWTORUSR1=%lgR2=%lg",&a, &b)!=2) error("Erwartet: new Torus r1=<r1> r2=<r2>");
        newobject=new TTorus(a,b);
      }
      else if (testcommand(str2,"CUBE"))
        newobject=new TCube;
      else if (testcommand(str2,"QUAD"))
      {
        if (testcommand(str2,"QUADXYF")) newobject=new TSimpleQuad(ebene_xyv);
        else if (testcommand(str2,"QUADXYB")) newobject=new TSimpleQuad(ebene_xyh);
        else if (testcommand(str2,"QUADXZT")) newobject=new TSimpleQuad(ebene_xzo);
        else if (testcommand(str2,"QUADXZB")) newobject=new TSimpleQuad(ebene_xzu);
        else if (testcommand(str2,"QUADYZL")) newobject=new TSimpleQuad(ebene_yzl);
        else if (testcommand(str2,"QUADYZR")) newobject=new TSimpleQuad(ebene_yzr);
        else error("Erwartet: new Quad xyf|xyb|xzt|xzb|yzl|yzr    (Ebene+Richtung)");
      }
      else if (testcommand(str2,"POLYGON"))
      {
        if (sscanf(str2,"POLYGONPOINTS=%d",&n)!=1)
          error("Erwartet: Polygon Points=<n>");
        if (n<3)
          error("Polygone mssen mindestens 3 Punkte haben!");
        newobject=new TPolygon;
        if (newobject==NULL) error ("Nicht gengend freier Speicher");
        for (i=0;i<n;++i)
        {
          ++line;
          fgets(str,255,datei);normalize(str,str3);
          if (sscanf(str3,"%lg,%lg,%lg",&p.x,&p.y,&p.z)!=3) error("Erwartet: <x>,<y>,<z>");
          ((TPolygon *)newobject)->addpoint(p);
        }
      }
      else if (testcommand(str2,"EQUATION"))
      {
        if (sscanf(str2,"EQUATIONCOEFFS=%d",&n)!=1)
          error("Erwartet: Equation Coeffs=<n>");
        newobject=new TEqnObject;
        if (newobject==NULL) error("Nicht gengend freier Speicher");
        for (i=0;i<n;++i)
        {
          ++line;
          fgets(str,255,datei);normalize(str,str3);
          if (sscanf(str3,"%lg,%d,%d,%d",&a,&j,&k,&l)!=4)
            error("Erwartet: <a>,<n>,<m>,<o>");
          ((TEqnObject*)newobject)->AddKoeff(a,j,k,l);
        }
      }
      else if (testcommand(str2,"SORSEG"))
      {
        if (sscanf(str2,"SORSEGA=%lgB=%lgC=%lgD=%lg",&a, &b, &c, &d)!=4)
          error("Erwartet: SORSEG a=<a> b=<b> c=<c> d=<d>");
        newobject=new TSorSegment(a,b,c,d);
      }
      else if (testcommand(str2,"SOR"))
      {
        if (sscanf(str2,"SORPOINTS=%d",&n)!=1)
          error("Erwartet: Sor Points=<n>");
        if (n<4)
          error("Ein Surface of Revolution bentigt mindestens 4 Punkte!");

        double *xlist, *ylist; xlist=new double[n]; ylist=new double[n];
        for (i=0;i<n;++i)
        {
          ++line;
          fgets(str,255,datei);normalize(str,str3);
          if (sscanf(str3,"%lg,%lg",xlist+i,ylist+i)!=2) error("Erwartet: <x>,<y>");
        }
        newobject=new TSor(n, xlist, ylist);
        delete xlist;
        delete ylist;        
      }
      else if (testcommand(str2,"BLOB"))
      {
        if (sscanf(str2,"BLOBBORDER=%lg",&a)!=1)
          error("Erwartet: New Blob border=<a>");
        newobject=new TBlob(a);
        if (newobject==NULL) error("Nicht gengend freier Speicher");
        ++line;
        fgets(str,255,datei);normalize(str,str2);
        while (!testcommand(str2,"END_BLOB"))
        {
          if (testcommand(str2,"SPHERICAL"))
          {
            if (sscanf(str2,"SPHERICALPOS=%lg,%lg,%lgINTENS=%lg",&a,&b,&c,&d)!=4)
              error("Erwartet: Spherical pos=<x>,<y>,<z> intens=<a>");
            ((TBlob*)newobject)->AddSpherical(a,b,c,d);
          }
          else if (testcommand(str2,"CYLINDRICAL"))
          {
            if (sscanf(str2,"CYLINDRICALP1=%lg,%lg,%lgP2=%lg,%lg,%lgINTENS=%lg",&a,&b,&c,&d,&e,&f,&g)!=7)
              error("Erwartet: Cylindrical P1=<x>,<y>,<z> P2=<x>,<y>,<z> intens=<a>");
            p.x=a;p.y=b;p.z=c;p2.x=d;p2.y=e;p2.z=f;
            ((TBlob*)newobject)->AddCylindrical(p,p2,g);
          }
          else if (testcommand(str2,"PLANAR"))
          {
            if (sscanf(str2,"PLANARP=%lg,%lg,%lgN=%lg,%lg,%lgINTENS=%lg",&a,&b,&c,&d,&e,&f,&g)!=7)
              error("Erwartet: Planar P=<x>,<y>,<z> N=<x>,<y>,<z> intens=<a>");
            p.x=a;p.y=b;p.z=c;p2.x=d;p2.y=e;p2.z=f;
            ((TBlob*)newobject)->AddPlanar(p,p2,g);
          }
          else error("Unbekanntes Blob-Objekt");
          if (debuglevel>=3) MessageFunc(line, "Blob-Objekt hinzugefgt");
          ++line;
          fgets(str,255,datei);normalize(str,str2);
        }
        if (debuglevel>=1) MessageFunc(line, "Berechne Blobfunktion...");
        ((TBlob*)newobject)->calculate();
      }
      else if (testcommand(str2,"METABALL"))
      {
        if (sscanf(str2,"METABALLBORDER=%lg",&a)!=1)
          error("Erwartet: New Metaball border=<a>");
        newobject=new TMetaBall(a);
        if (newobject==NULL) error("Nicht gengend freier Speicher");
        ++line;
        fgets(str,255,datei);normalize(str,str2);
        while (!testcommand(str2,"END_METABALL"))
        {
          if (testcommand(str2,"SPHERICAL"))
          {
            if (sscanf(str2,"SPHERICALPOS=%lg,%lg,%lgINTENS=%lgRAD=%lg",&a,&b,&c,&d,&e)!=5)
              error("Erwartet: Spherical pos=<x>,<y>,<z> intens=<a> rad=<r>");
            ((TMetaBall*)newobject)->AddSpherical(a,b,c,d,e);
          }
          else error("Unbekanntes Metaball-Objekt");
          if (debuglevel>=3) MessageFunc(line, "Metaball-Objekt hinzugefgt");
          ++line;
          fgets(str,255,datei);normalize(str,str2);
        }
      }
      else if (testcommand(str2,"ROTOBJ"))
      {
        if (sscanf(str2,"ROTOBJPOINTS=%d",&n)!=1)
          error("Erwartet: Rotobj Points=<n>");
        if (n<2)
          error("Rotationskrper mssen mindestens 2 Punkte haben!");
        newobject=new TRotobj;
        if (newobject==NULL) error ("Nicht gengend freier Speicher");
        for (i=0;i<n;++i)
        {
          ++line;
          fgets(str,255,datei);normalize(str,str3);
          if (sscanf(str3,"%lg,%lg",&a,&b)!=2) error("Erwartet: <x>,<y>");
          ((TRotobj *)newobject)->add(a,b);
          if (debuglevel>=3) MessageFunc(line, "Rotationskrper-Punkt hinzugefgt");
        }
      }
      else if (testcommand(str2,"CONE"))
      {
        if (sscanf(str2,"CONEP1=%lg,%lgP2=%lg,%lg",&a,&b,&c,&d)!=4)
          error("Erwartet: CONE P1=<x>,<y> P2=<x>,<y>");
        newobject=new TCone(a,b,c,d);
      }
      else if (testcommand(str2,"DUST"))
      {
        if (sscanf(str2,"DUSTDENSITY=%lgREFLECT=%lg",&a,&b)!=2)
          error("Erwartet: DUST density=<a> reflect=<b>");
        newobject=new TDust(a,b);
      }
      else if (testcommand(str2,"3DPLOT"))
      {
        if (sscanf(str2,"3DPLOT%d*%dX1=%lgY1=%lgX2=%lgY2=%lg",&j,&k,&a,&b,&c,&d)!=6)
          error("Erwartet: 3DPlot <x>*<y> x1=<x1> y1=<y1> x2=<x2> y2=<y2>");
        ++line;
        fgets(str,255,datei);normalize(str,str3);
        if (debuglevel>=1) MessageFunc(line, "Erstelle Funktionsplot...");
        newobject=new T3dPlot(j,k,str3,a,b,c,d,l);
        if (!l)
          error("Fehler in dem Funktionsterm");
      }
      else if (testcommand(str2,"LANDSCAPE"))
      {
        if (sscanf(str2,"LANDSCAPESIZE=%dFRACDIM=%lgDELTA=%lgRAND=%dP1=%lgP2=%lgP3=%lgP4=%lg",&k,&a,&b,&l,&c,&d,&e,&f)!=8)
          error("Erwartet: Landscape Size=<n> Fracdim=<a> Delta=<b> Rand=<m> P1=<h1> P2=<h2> P3=<h3> P4=<h4>");
        if (debuglevel>=1) MessageFunc(line, "Erstelle fraktale Landschaft...");
        newobject=new TLandscape(k,a,b,l,c,d,e,f);
      }
      else if (testcommand(str2,"HEIGHTFIELD"))
      {
        if (sscanf(str2,"HEIGHTFIELD%d*%d%s",&j,&k,str3)!=3)
          error("Erwartet: Heightfield <x>*<y> Dateiname.BMP");
        newobject=new TBMPHeightField(j,k,str3,l);
        if (!l)
          error("Bitmap nicht gefunden oder falsches Dateiformat");
      }
      else error("Unbekannter Objekttyp");
      if (newobject==NULL) error ("Nicht gengend freier Speicher");
      if (newobject->type==TYPE_DUST)
      {
        newobject->reflect=0;
        newobject->light=0;
        newobject->brech=1;
      }
      else
      {
        newobject->reflect=curdata.reflect;
        newobject->trans=curdata.trans;
        newobject->flat=curdata.flat;
        newobject->light=curdata.light;
        newobject->brech=curdata.brech;
        newobject->phongsize=curdata.phongsize;
      }
      if (debuglevel>=2)
      {
        sprintf(messagestr, "Objekt vom Typ %d hinzugefgt", newobject->type);
        MessageFunc(line, messagestr);
      }
      if (debuglevel>=3)
      {
        sprintf(messagestr, "   (reflect=%lg, trans=%lg, flat=%lg, light=%lg, brech=%lg, phongsize=%lg)",
          newobject->reflect, newobject->trans, newobject->flat, newobject->light, newobject->brech, newobject->phongsize);
        MessageFunc(line, messagestr);
      }



    }

/*------------------------------Objektattribute-----------------------------*/

    else if (testcommand(command,"COLOR"))
    {
      if (newobject==NULL) error("Color darf nur in Objekten benutzt werden");
      if (sscanf(command,"COLOR=%d,%d,%d",&cr,&cg,&cb)!=3)
        error("Erwartet: Color=<r>,<g>,<b>         r,g,b=0..255");
      if (cr>255 || cg>255 || cb>255 || cr<0 || cg<0 || cb<0)
        error("Erwartet: Color=<r>,<g>,<b>         r,g,b=0..255");
      newobject->clrred=cr;
      newobject->clrgreen=cg;
      newobject->clrblue=cb;
      colordef=1;
    }

    else if (testcommand(command,"NO_DITHER"))
    {
      if (newobject==NULL && containernum==0) error("NO_DITHER darf nur in Objekten oder Containern benutzt werden");
      if (newobject!=NULL)
        newobject->nodither=1;
      else
        Container[containernum-1]->nodither=1;
    }
    else if (testcommand(command,"ROTATED"))
    {
      if (newobject==NULL && containernum==0) error("Rotated darf nur in Objekten oder Containern benutzt werden");
      if (sscanf(command,"ROTATED%lg,%lg,%lg",&a,&b,&c)!=3)
        error("Erwartet: Rotated <ax>,<ay>,<az>");
      if (newobject!=NULL)
      {
        newobject->rotatex=a;
        newobject->rotatey=b;
        newobject->rotatez=c;
      }
      else
      {
        Container[containernum-1]->rotatex=a;
        Container[containernum-1]->rotatey=b;
        Container[containernum-1]->rotatez=c;
      }
    }
    else if (testcommand(command,"SCALED"))
    {
      if (newobject==NULL && containernum==0) error("Scaled darf nur in Objekten oder Containern benutzt werden");
      if (sscanf(command,"SCALED%lg,%lg,%lg",&a,&b,&c)!=3)
        error("Erwartet: Scaled <fx>,<fy>,<fz>");
      if (newobject!=NULL)
      {
        newobject->scalex=a;
        newobject->scaley=b;
        newobject->scalez=c;
      }
      else
      {
        Container[containernum-1]->scalex=a;
        Container[containernum-1]->scaley=b;
        Container[containernum-1]->scalez=c;
      }
    }
    else if (testcommand(command,"POS"))
    {
      if (newobject==NULL && containernum==0) error("Pos darf nur in Objekten oder Containern benutzt werden");
      if (sscanf(command,"POS=%lg,%lg,%lg",&a,&b,&c)!=3)
        error("Erwartet: Pos=<fx>,<fy>,<fz>");
      if (newobject!=NULL)
      {
        newobject->midx=a;
        newobject->midy=b;
        newobject->midz=c;
      }
      else
      {
        Container[containernum-1]->midx=a;
        Container[containernum-1]->midy=b;
        Container[containernum-1]->midz=c;
      }
    }

/*-------------------------Ende neues Objekt--------------------------------*/

    else if (testcommand(command,"END_OBJECT"))
    {
      if (!colordef) error ("Es wurde keine Objektfarbe angegeben!");
      if (newobject==NULL) error("End_Object ohne Objektdefinition");
      if (containernum==0)
      {
        TheTracer->addobject(newobject);
        if (debuglevel>=3)
          MessageFunc(line, "  Objekt wurde dem Raytracer hinzugefgt");
      }
      else
      {
        switch(Container[containernum-1]->type)
        {
          case TYPE_UNION:
            ((TUnion *)Container[containernum-1])->addobject(newobject);
            break;
          case TYPE_CONTAINER:
            ((TContainer *)Container[containernum-1])->addobject(newobject);
            break;
          case TYPE_TEXTURE:
            ((TTexture *)Container[containernum-1])->addobject(newobject);
            break;
          case TYPE_BUMPMAP:
            ((TBumpMap *)Container[containernum-1])->addobject(newobject);
            break;
          case TYPE_INTERSECTION:
            ((TIntersection *)Container[containernum-1])->addobject(newobject);
            break;
          case TYPE_SUBTRACTION:
            if (++((TSubtraction *)Container[containernum-1])->assignednum==1)
              ((TSubtraction *)Container[containernum-1])->Obja=newobject;
            else if (((TSubtraction *)Container[containernum-1])->assignednum==2)
              ((TSubtraction *)Container[containernum-1])->Objb=newobject;
            else error("Subtraktionskrper knnen nur zwei Elemente (Minuend und Subtrahend) haben");
            break;
        }
        if (debuglevel>=3)
          MessageFunc(line, "  Objekt wurde dem Container hinzugefgt");
      }
      newobject=NULL;
    }

/*------------------------------Neue Container------------------------------*/

    else if (testcommand(command,"UNION"))
    {
      if (++containernum>COMPMAXREKUR)
        error("Die Verschachtelungstiefe ist zu gro!");
      Container[containernum-1]=new TUnion();
      if (Container[containernum-1]==NULL) error ("Nicht gengend freier Speicher");
      if (debuglevel>=2)
      {
        sprintf(messagestr, "Neuer Container (Union), Verschachtelungstiefe=%d", containernum);
        MessageFunc(line, messagestr);
      }
    }
    else if (testcommand(command,"CONTAINER"))
    {
      if (++containernum>COMPMAXREKUR)
        error("Die Verschachtelungstiefe ist zu gro!");
      Container[containernum-1]=new TContainer();
      if (Container[containernum-1]==NULL) error ("Nicht gengend freier Speicher");
      if (debuglevel>=2)
      {
        sprintf(messagestr, "Neuer Container (Container), Verschachtelungstiefe=%d", containernum);
        MessageFunc(line, messagestr);
      }
    }
    else if (testcommand(command,"INTERSECTION"))
    {
      if (++containernum>COMPMAXREKUR)
        error("Die Verschachtelungstiefe ist zu gro!");
      Container[containernum-1]=new TIntersection();
      if (Container[containernum-1]==NULL) error ("Nicht gengend freier Speicher");
      if (debuglevel>=2)
      {
        sprintf(messagestr, "Neuer Container (Intersection), Verschachtelungstiefe=%d", containernum);
        MessageFunc(line, messagestr);
      }
    }
    else if (testcommand(command,"SUBTRACTION"))
    {
      if (++containernum>COMPMAXREKUR)
        error("Die Verschachtelungstiefe ist zu gro!");
      Container[containernum-1]=new TSubtraction();
      if (Container[containernum-1]==NULL) error ("Nicht gengend freier Speicher");
      if (debuglevel>=2)
      {
        sprintf(messagestr, "Neuer Container (Subtraction), Verschachtelungstiefe=%d", containernum);
        MessageFunc(line, messagestr);
      }
    }
    else if (testcommand(command,"END_CONTAINER"))
    {
      if (--containernum<0)
        error("END_CONTAINER ohne Containerdefinition (Union,Intersection,Subtraction)");
      if (Container[containernum]->type==TYPE_TEXTURE)
      {
        istexture=0;
        if (((TTexture*)Container[containernum])->BaseAlgo==NULL)
          error("Es wurde kein Basisalgorithmus fr die Textur angegeben");
      }
      if (Container[containernum]->type==TYPE_BUMPMAP)
      {
        istexture=0;
        if (((TBumpMap*)Container[containernum])->BaseAlgo==NULL)
          error("Es wurde kein Basisalgorithmus fr das Bumpmap angegeben");
      }
      if (containernum==0)
      {
        TheTracer->addobject(Container[containernum]);
        if (debuglevel>=3) MessageFunc(line, "  Container wurde dem Raytracer hinzugefgt");
      }
      else
      {
        switch(Container[containernum-1]->type)
        {
          case TYPE_TEXTURE:
          case TYPE_BUMPMAP:
          case TYPE_INTERSECTION:
          case TYPE_UNION:
          case TYPE_CONTAINER:
            ((TIntersection *)Container[containernum-1])->addobject(Container[containernum]);
            break;
          case TYPE_SUBTRACTION:
            if (++((TSubtraction *)Container[containernum-1])->assignednum==1)
              ((TSubtraction *)Container[containernum-1])->Obja=Container[containernum];
            else if (((TSubtraction *)Container[containernum-1])->assignednum==2)
              ((TSubtraction *)Container[containernum-1])->Objb=Container[containernum];
            else error("Subtraktionskrper knnen nur zwei Elemente (Minuend und Subtrahend) haben");
            break;
        }
        if (debuglevel>=3) MessageFunc(line, "  Container wurde bergeordnetem Container hinzugefgt");
      }
    }

/*-------------------------Texturen&-Attribute------------------------------*/

    else if (testcommand(command,"TEXTURE"))
    {
      if (++containernum>COMPMAXREKUR)
        error("Die Verschachtelungstiefe ist zu gro!");
      if (istexture)
        error("Es darf immer nur eine Textur gleichzeitig verwendet werden");
      Container[containernum-1]=new TTexture();
      if (Container[containernum-1]==NULL) error ("Nicht gengend freier Speicher");
      istexture=1;
      if (debuglevel>=2)
      {
        sprintf(messagestr, "Neuer Container (Texture), Verschachtelungstiefe=%d", containernum);
        MessageFunc(line, messagestr);
      }
    }
    else if (testcommand(command,"BASE"))
    {
      if (!istexture) error("BASE darf nur in Texturdefinitionen vorkommen");
      if (((TTexture*)Container[containernum-1])->BaseAlgo!=NULL)
        error("Es darf nur ein Basisalgorithmus angegeben werden");
      if (testcommand(command,"BASECUBES"))
      {
        if (sscanf(command,"BASECUBESCLR1=%d,%d,%dCLR2=%d,%d,%d",&cr,&cg,&cb,&j,&k,&l)!=6)
          error("Erwartet: BASE Cubes Clr1=<r1>,<g1>,<b1> Clr2=<r2>,<g2>,<b2>");
        ((TTexture*)Container[containernum-1])->BaseAlgo=new TText_Cubes(cr,cg,cb,j,k,l);
      }
      else if (testcommand(command,"BASEWOOD"))
      {
        if (sscanf(command,"BASEWOODCLR1=%d,%d,%dCLR2=%d,%d,%d",&cr,&cg,&cb,&j,&k,&l)!=6)
          error("Erwartet: BASE Wood Clr1=<r1>,<g1>,<b1> Clr2=<r2>,<g2>,<b2>");
        ((TTexture*)Container[containernum-1])->BaseAlgo=new TText_Wood(cr,cg,cb,j,k,l);
      }
      else if (testcommand(command,"BASEMARBLE"))
      {
        if (sscanf(command,"BASEMARBLE%lgCLR1=%d,%d,%dCLR2=%d,%d,%d",&a,&cr,&cg,&cb,&j,&k,&l)!=7)
        {
          a=1;
          if (sscanf(command,"BASEMARBLECLR1=%d,%d,%dCLR2=%d,%d,%d",&cr,&cg,&cb,&j,&k,&l)!=6)
          error("Erwartet: BASE Marble [<a>] Clr1=<r1>,<g1>,<b1> Clr2=<r2>,<g2>,<b2>");
        }
        ((TTexture*)Container[containernum-1])->BaseAlgo=new TText_Marble(cr,cg,cb,j,k,l,a);
      }
      else if (testcommand(command,"BASESTAR"))
      {
        if (sscanf(command,"BASESTARR1=%lfR2=%lf",&a,&b)!=2)
          error("Erwartet: BASE Star r1=<a> r2=<b>");
        ((TTexture*)Container[containernum-1])->BaseAlgo=new TText_Star(a,b);
      }
      else if (testcommand(command,"BASECLOUD"))
      {
        if (sscanf(command,"BASECLOUD%lg",&a)!=1)
          error("Erwartet: Base Cloud <p>");
        ((TTexture*)Container[containernum-1])->BaseAlgo=new TText_Cloud(a);
      }
      else if (testcommand(command,"BASELANDSCAPE"))
      {
        if (sscanf(command,"BASELANDSCAPECOLORS=%d",&n)!=1)
          error("Erwartet: BASE Landscape Colors=<n>");
        if (n<1)
          error("Es mu mindestens ein Farbverlauf angegeben werden!");
        ((TTexture*)Container[containernum-1])->BaseAlgo=new TText_Landscape();
        for (i=0;i<n;++i)
        {
          ++line;
          fgets(str,255,datei);normalize(str,str3);
          if (sscanf(str3,"P1=%lgP2=%lgCLR1=%d,%d,%dCLR2=%d,%d,%d",&a,&b,&cr,&cg,&cb,&j,&k,&l)!=8)
            error("Erwartet: P1=<a> P2=<b CLR1=<r>,<g>,<b> CLR2=<r>,<g>,<b>");
          ((TText_Landscape*)(((TTexture*)Container[containernum-1])->BaseAlgo))->AddColors(cr,cg,cb,j,k,l,a,b);
        }
      }
      else if (testcommand(command,"BASEMAPXY"))
      {
        extractfilename(str, str3); if (strlen(str3)<1)
          //        if (sscanf(command,"BASEMAPXY%s",str3)!=1)
          error("Erwartet: BASE MapXY Dateiname.BMP");
        ((TTexture*)Container[containernum-1])->BaseAlgo=new TText_MapBmpXY(str3);
        if (((TText_MapBmpXY *)(((TTexture*)Container[containernum-1])->BaseAlgo))->Bmp==NULL)
          error("Bitmap nicht gefunden oder falsches Dateiformat");
      }
      else if (testcommand(command,"BASEMAPCYLINDRICAL"))
      {
        if (sscanf(command,"BASEMAPCYLINDRICAL%s",str3)!=1)
          error("Erwartet: BASE MapCylindrical Dateiname.BMP");
        ((TTexture*)Container[containernum-1])->BaseAlgo=new TText_MapBmpCylindrical(str3);
        if (((TText_MapBmpCylindrical *)(((TTexture*)Container[containernum-1])->BaseAlgo))->Bmp==NULL)
          error("Bitmap nicht gefunden oder falsches Dateiformat");
      }
      else if (testcommand(command,"BASEMAPSPHERICAL"))
      {
        if (sscanf(command,"BASEMAPSPHERICAL%s",str3)!=1)
          error("Erwartet: BASE MapSpherical Dateiname.BMP");
        ((TTexture*)Container[containernum-1])->BaseAlgo=new TText_MapBmpSpherical(str3);
        if (((TText_MapBmpSpherical *)(((TTexture*)Container[containernum-1])->BaseAlgo))->Bmp==NULL)
          error("Bitmap nicht gefunden oder falsches Dateiformat");
      }
      else error("Unbekannter Textur-Basisalgorithmus!");
      if (debuglevel>=3) MessageFunc(line, "  Textur-Basisalgorithmus hinzugefgt");
    }
    else if (testcommand(command,"BUMPMAP"))
    {
      if (++containernum>COMPMAXREKUR)
        error("Die Verschachtelungstiefe ist zu gro!");
      Container[containernum-1]=new TBumpMap();
      if (Container[containernum-1]==NULL) error ("Nicht gengend freier Speicher");
      if (debuglevel>=2)
      {
        sprintf(messagestr, "Neuer Container (Bumpmap), Verschachtelungstiefe=%d", containernum);
        MessageFunc(line, messagestr);
      }
    }
    else if (testcommand(command,"BUMPBASE"))
    {
      if (Container[containernum-1]->type!=TYPE_BUMPMAP)
        error("BUMPBASE darf nur in Bumpmap-Definitionen vorkommen!");
      if (testcommand(command,"BUMPBASERINGS"))
      {
        if (sscanf(command,"BUMPBASERINGSRAD=%lgINTENS=%lgDECREASE=%lg",&a,&b,&c)!=3)
          error("Erwartet: BUMPBASE RINGS rad=<a> intens=<b> decrease=<c>");
        ((TBumpMap*)Container[containernum-1])->BaseAlgo=new TBump_Rings(a,b,c);
      }
      else if (testcommand(command,"BUMPBASESINES"))
      {
        if (sscanf(command,"BUMPBASESINESINTENS=%lg",&a)!=1)
          error("Erwartet: BUMPBASE SINES INTENS=<a>");
        ((TBumpMap*)Container[containernum-1])->BaseAlgo=new TBump_Sines(a);
      }
      else error("Unbekannter Bumpmap-Basisalgorithmus!");
      if (debuglevel>=3) MessageFunc(line, "  Bumpmap-Basisalgorithmus hinzugefgt");
    }
    else if (testcommand(command,"MODIFIERS"))
    {
      if (Container[containernum-1]->type!=TYPE_TEXTURE
        &&Container[containernum-1]->type!=TYPE_BUMPMAP)
        error("MODIFIERS darf nur in Textur- oder Bumpmapdefinitionen vorkommen");
      do
      {
        ++line;
        fgets(str,255,datei);normalize(str,str3);
        if (testcommand(str3,"END_MODIFIERS")) continue;
        if (testcommand(str3,"SCALE"))
        {
          if (sscanf(str3,"SCALE%lg,%lg,%lg",&a,&b,&c)!=3)
             error("Erwartet: SCALE <ax>,<ay>,<az>");
          if (Container[containernum-1]->type==TYPE_TEXTURE)
            ((TTexture*)Container[containernum-1])->addmodifier(new TMod_Scale(a,b,c));
          else
            ((TBumpMap*)Container[containernum-1])->addmodifier(new TMod_Scale(a,b,c));
        }
        if (testcommand(str3,"TRANSLATE"))
        {
          if (sscanf(str3,"TRANSLATE%lg,%lg,%lg",&a,&b,&c)!=3)
             error("Erwartet: TRANSLATE <ax>,<ay>,<az>");
          if (Container[containernum-1]->type==TYPE_TEXTURE)
            ((TTexture*)Container[containernum-1])->addmodifier(new TMod_Translate(a,b,c));
          else
            ((TBumpMap*)Container[containernum-1])->addmodifier(new TMod_Translate(a,b,c));
        }
        if (testcommand(str3,"ROTATE"))
        {
          if (sscanf(str3,"ROTATE%lg,%lg,%lg",&a,&b,&c)!=3)
             error("Erwartet: ROTATE <ax>,<ay>,<az>");
          if (Container[containernum-1]->type==TYPE_TEXTURE)
            ((TTexture*)Container[containernum-1])->addmodifier(new TMod_Rotate(a,b,c));
          else
            ((TBumpMap*)Container[containernum-1])->addmodifier(new TMod_Rotate(a,b,c));
        }
        if (testcommand(str3,"NOISE"))
        {
          if (sscanf(str3,"NOISE%d,%lg",&i,&a)!=2)
            error("Erwartet: NOISE <n>,<a>");
          if (Container[containernum-1]->type==TYPE_TEXTURE)
            ((TTexture*)Container[containernum-1])->addmodifier(new TMod_Noise(i,a));
          else
            ((TBumpMap*)Container[containernum-1])->addmodifier(new TMod_Noise(i,a));
        }
        if (testcommand(str3,"TURBULENCE"))
        {
          if (sscanf(str3,"TURBULENCE%d,%lg",&i,&a)!=2)
            error("Erwartet: TURBULENCE <n>,<a>");
          if (Container[containernum-1]->type==TYPE_TEXTURE)
            ((TTexture*)Container[containernum-1])->addmodifier(new TMod_Turbulence(i,a));
          else
            ((TBumpMap*)Container[containernum-1])->addmodifier(new TMod_Turbulence(i,a));
        }
        if (debuglevel>=3) MessageFunc(line, "  Modifier hinzugefgt");
      } while (!testcommand(str3,"END_MODIFIERS"));
    }

/*---------------------------Kameradefinition-------------------------------*/

    else if (testcommand(command,"CAMERA"))
    {
      if (cameradef) error("Es kann nur eine Kamera angegeben werden!");
      if (sscanf(command,"CAMERAFROM=%lg,%lg,%lgTO=%lg,%lg,%lgSIZE=%lg*%lg",&a,&b,&c,&d,&e,&f,&g,&h)!=8)
        error("Erwartet: Camera From=<x>,<y>,<z> To=<x>,<y>,<u> Size=<x>,<y>");
      TheTracer->Camera=new TNormalCamera(a,b,c,d,e,f,resx,resy,g,h);
      if (TheTracer->Camera==NULL) error ("Nicht gengend freier Speicher");
      cameradef=1;
      if (debuglevel>=3) MessageFunc(line, "Kamera hinzugefgt");
    }

/*---------------------------Materialdefinition-----------------------------*/

    else if (testcommand(command,"MATERIAL"))
      ismaterial=1;
    else if (testcommand(command,"FLAT"))
    {
      if (!ismaterial) error("Flat kann nur in Materialdefinitionen verwendet werden");
      if (sscanf(command,"FLAT=%lg",&a)!=1) error("Erwartet: Flat=<f>");
      curdata.flat=a;
    }
    else if (testcommand(command,"LIGHT") && ismaterial)  //Es gibt auch Lichtquellen!
    {
      if (!ismaterial) error("Light kann nur in Materialdefinitionen verwendet werden");
      if (sscanf(command,"LIGHT=%lg",&a)!=1) error("Erwartet: Light=<f>");
      curdata.light=a;
    }
    else if (testcommand(command,"TRANS"))
    {
      if (!ismaterial) error("Trans kann nur in Materialdefinitionen verwendet werden");
      if (sscanf(command,"TRANS=%lg",&a)!=1) error("Erwartet: Trans=<f>");
      curdata.trans=a;
    }
    else if (testcommand(command,"REFLECT"))
    {
      if (!ismaterial) error("Reflect kann nur in Materialdefinitionen verwendet werden");
      if (sscanf(command,"REFLECT=%lg",&a)!=1) error("Erwartet: Reflect=<f>");
      curdata.reflect=a;
    }
    else if (testcommand(command,"FRACTION"))
    {
      if (!ismaterial) error("Fraction kann nur in Materialdefinitionen verwendet werden");
      if (sscanf(command,"FRACTION=%lg",&a)!=1) error("Erwartet: Fraction=<f>");
      curdata.brech=a;
    }
    else if (testcommand(command,"PHONGSIZE"))
    {
      if (!ismaterial) error("Phongsize kann nur in Materialdefinitionen verwendet werden");
      if (sscanf(command,"PHONGSIZE=%lg",&a)!=1) error("Erwartet: Phongsize=<f>");
      curdata.phongsize=a;
    }
    else if (testcommand(command,"END_MATERIAL")) ismaterial=0;

/*---------------------------Lichtquellen-----------------------------------*/

    else if (testcommand(command,"LIGHTTYPE=OMNI"))
    {
      if (sscanf(command,"LIGHTTYPE=OMNIPOS=%lg,%lg,%lgCOLOR=%d,%d,%dINTENSITY=%lg",&a,&b,&c,&cr,&cg,&cb,&d)!=7)
        error("Erwartet: Light type=Omni pos=<x>,<y>,<z> color=<r>,<g>,<b> Intensity=<i>");
      if (cr<0||cr>255||cg<0||cg>255||cb<0||cb>255)
        error("Farbwerge mssen zwischen 0 und 255 liegen");
      TheTracer->addlight(new TOmniLight(cr,cg,cb,d,a,b,c));
      if (TheTracer->Light[TheTracer->numlight-1]==NULL) error ("Nicht gengend freier Speicher");
      lightdef=1;
      if (debuglevel>=3)  MessageFunc(line, "Lichtquelle (Omni) hinzugefgt");
    }
    else if (testcommand(command,"LIGHTTYPE=SPOT"))
    {
      if (sscanf(command,"LIGHTTYPE=SPOTFROM=%lg,%lg,%lgTO=%lg,%lg,%lgANGLE=%lgCOLOR=%d,%d,%dINTENSITY=%lg",&a,&b,&c,&d,&e,&f,&g,&cr,&cg,&cb,&h)!=11)
        error("Erwartet: Light type=Spot from=<x>,<y>,<z> to=<x>,<y>,<z>\r\nangle=<a> color=<r>,<g>,<b> Intensity=<i>");
      if (cr<0||cr>255||cg<0||cg>255||cb<0||cb>255)
        error("Farbwerte mssen zwischen 0 und 255 liegen");
      TheTracer->addlight(new TSpotLight(cr,cg,cb,h,a,b,c,d,e,f,g));
      if (TheTracer->Light[TheTracer->numlight-1]==NULL) error ("Nicht gengend freier Speicher");
      lightdef=1;
      if (debuglevel>=3)  MessageFunc(line, "Lichtquelle (Spot) hinzugefgt");
    }
    else if (testcommand(command,"LIGHTTYPE=AMBIENT"))
    {
      if (sscanf(command,"LIGHTTYPE=AMBIENTCOLOR=%d,%d,%d",&cr,&cg,&cb)!=3)
        error("Erwartet: Light type=Ambient Color=<r>,<g>,<b>");
      if (cr<0||cr>255||cg<0||cg>255||cb<0||cb>255)
        error("Farbwerte mssen zwischen 0 und 255 liegen");
      TheTracer->addlight(new TAmbientLight(cr,cg,cb));
      if (TheTracer->Light[TheTracer->numlight-1]==NULL) error ("Nicht gengend freier Speicher");
      lightdef=1;
      if (debuglevel>=3)  MessageFunc(line, "Lichtquelle (Ambient) hinzugefgt");
    }

    else if (strlen(command)!=0) error("Unbekannter Befehl");
  }
  if (!cameradef) error("Es wurde keine Kamera angegeben!");
  if (!lightdef) error("Es wurde keine Lichtquelle angegeben!");
  fclose(datei);
  if (debuglevel>=3) MessageFunc(line, "Script-Datei geschlossen");
  return 1;
}

#pragma warn .def
