/****************************************************************************/
/* TEXTBASE.CPP - Texturbasisalgorithmen                                    */
/*  Copyright   (c) 1999 Martin Melcher                                     */
/* martin@raytracer.de      Gluckstr. 24/42655 Solingen/Germany             */
/* This program is free software; you can redistribute it and/or modify it  */
/* under the terms of the GNU General Public License as published by the    */
/* Free Software Foundation; either version 2 of the License, or (at your   */
/* option) any later version.                                               */
/* This Program is distributed in the hope that it will be useful, but      */
/* WITHOUT ANY WARRANTY; without even the implied warranty of               */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPUOSE. See the GNU General*/
/* Public License for more details.                                         */
/* You should have received a copy of the GNU General Publiv License along  */
/* with this program; if not, write to the Free Software Foundation, Inc.,  */
/* 675 Mass Ave, Cambridge, MA 02139, USA.                                  */
/****************************************************************************/

#include<math.h>
#include"textbase.h"
#include"readbmp.h"

void TText_Cubes::GetSurface(TVector const &point, TObjectData *surface)
{
  long int x,y,z,s;
  x=point.x;y=point.y;z=point.z;
  if (point.x<0) --x;
  if (point.y<0) --y;
  if (point.z<0) --z;
  s=x+y+z;
  if (s<0) s=-s;
  s%=2;
  surface->clrred*=s*r1+(1-s)*r2;
  surface->clrgreen*=s*g1+(1-s)*g2;
  surface->clrblue*=s*b1+(1-s)*b2;
  surface->clrred/=256;
  surface->clrgreen/=256;
  surface->clrblue/=256;
}


void TText_Wood::GetSurface(TVector const &point, TObjectData *surface)
{
  double dist,s;
  dist=sqrt(point.x*point.x+point.z*point.z);
  s=dist-(int)dist;
  surface->clrred*=s*r1+(1-s)*r2;
  surface->clrgreen*=s*g1+(1-s)*g2;
  surface->clrblue*=s*b1+(1-s)*b2;
  surface->clrred/=256;
  surface->clrgreen/=256;
  surface->clrblue/=256;
}

void TText_Marble::GetSurface(TVector const & point, TObjectData *surface)
{
  double s;
  s=(sin(point.x)+1.0)/2.0;
  s=pow(s,pot);
  surface->clrred*=s*r1+(1-s)*r2;
  surface->clrgreen*=s*g1+(1-s)*g2;
  surface->clrblue*=s*b1+(1-s)*b2;
  surface->clrred/=256;
  surface->clrgreen/=256;
  surface->clrblue/=256;
}

void TText_Star::GetSurface(TVector const &point, TObjectData *surface)
{
  double r;
  r=(double)point;
  if (r<r1) //Im Innern: Nicht verndern
    return;
  if (r>r2) //Auen : Ganz durchsichtig
  {
    surface->trans=1;
    surface->flat=0;
    return;
  }
  surface->trans=1-(1-surface->trans)*(r2-r)/(r2-r1);  //Dazwischen interpolieren
  surface->flat*=(r2-r)/(r2-r1);
}

void TText_Cloud::GetSurface(TVector const &point, TObjectData *surface)
{
  double t;
  t=(sin(point.x)+1.0)/2.0;
  t=pow(t,pot);
  surface->trans=1-(1-surface->trans)*t;
  surface->flat*=t;
}

void TText_Landscape::AddColors(char pr1, char pg1, char pb1, char pr2, char pg2, char pb2, double ppos1, double ppos2)
{
  ++num;
  r1=(char*)realloc(r1,num);
  g1=(char*)realloc(g1,num);
  b1=(char*)realloc(b1,num);
  r2=(char*)realloc(r2,num);
  g2=(char*)realloc(g2,num);
  b2=(char*)realloc(b2,num);
  pos1=(double*)realloc(pos1,num*sizeof(double));
  pos2=(double*)realloc(pos2,num*sizeof(double));
  r1[num-1]=pr1;
  g1[num-1]=pg1;
  b1[num-1]=pb1;
  r2[num-1]=pr2;
  g2[num-1]=pg2;
  b2[num-1]=pb2;
  pos1[num-1]=ppos1;
  pos2[num-1]=ppos2;
}

void TText_Landscape::GetSurface(TVector const &point, TObjectData *surface)
{
  int i;
  double factor;
  for (i=0;i<num;++i)
    if (pos1[i]<=point.y && pos2[i]>=point.y)
    {
      factor=(point.y-pos1[i])/(pos2[i]-pos1[i]);
      surface->clrred  =factor*r2[i]+(1.0-factor)*r1[i];
      surface->clrgreen=factor*g2[i]+(1.0-factor)*g1[i];
      surface->clrblue =factor*b2[i]+(1.0-factor)*b1[i];
      return;
    }
}

TText_MapBmpXY::TText_MapBmpXY(char *filename)
{
  Bmp=new TBMPInput(filename);
  if (Bmp->rgb==NULL) { delete Bmp; Bmp=NULL; return; }
}

TText_MapBmpXY::~TText_MapBmpXY()
{
  if (Bmp!=NULL)
  delete Bmp;
}

void TText_MapBmpXY::GetSurface(TVector const &point, TObjectData *surface)
{
  int r,g,b;
  int x,y;
  if (Bmp==NULL) return;
  x=(point.x+0.5)*Bmp->xsize;
  y=(point.y+0.5)*Bmp->ysize;
  x%=Bmp->xsize; if (x<0) x+=Bmp->xsize;
  y%=Bmp->ysize; if (y<0) y+=Bmp->ysize;
  Bmp->GetPixel(x,y, r,g,b);
  surface->clrred*=r;
  surface->clrgreen*=g;
  surface->clrblue*=b;
  surface->clrred/=256;
  surface->clrgreen/=256;
  surface->clrblue/=256;
}

void TText_MapBmpCylindrical::GetSurface(TVector const &point, TObjectData *surface)
{
  int r,g,b;
  int x,y;
  double angle;
  if (Bmp==NULL) return;
  y=(point.y+0.5)*Bmp->ysize;
  y%=Bmp->ysize; if (y<0) y+=Bmp->ysize;
  if ((fabs(point.x)<1e-5) && (fabs(point.z)<1e-5)) angle=0;
  else angle=atan2(point.z, point.x);
  x=(angle+M_PI)*Bmp->xsize/(M_PI*2.0);
  x%=Bmp->xsize; if (x<0) x+=Bmp->xsize;
  Bmp->GetPixel(x,y, r,g,b);
  surface->clrred*=r;
  surface->clrgreen*=g;
  surface->clrblue*=b;
  surface->clrred/=256;
  surface->clrgreen/=256;
  surface->clrblue/=256;
}

void TText_MapBmpSpherical::GetSurface(TVector const &point, TObjectData *surface)
{
  int r,g,b;
  int x,y;
  double angle1,angle2;
  TVector tmp(0,-1,0);
  if (Bmp==NULL) return;
  if ((fabs(point.x)<1e-5) && (fabs(point.z)<1e-5)) angle2=0;
  else  angle2=atan2(point.z, point.x);

  angle1=point.cosangle(tmp);
  if (angle1!=2) angle1=acos(angle1); else angle1=0;
  
  x=(angle2+M_PI)*Bmp->xsize/(M_PI*2.0);
  y=angle1*Bmp->ysize/M_PI;

  x%=Bmp->xsize; if (x<0) x+=Bmp->xsize;
  y%=Bmp->ysize; if (y<0) y+=Bmp->ysize;
  Bmp->GetPixel(x,y, r,g,b);
  surface->clrred*=r;
  surface->clrgreen*=g;
  surface->clrblue*=b;
  surface->clrred/=256;
  surface->clrgreen/=256;
  surface->clrblue/=256;
}
 
