/****************************************************************************/
/* 3D - Vektorrechnung fuer 3D-Reealtime und Raytracer                      */
/*      Basierend auf double                                                */
/*  Copyright   (c) 1999 Martin Melcher                                     */
/* martin@raytracer.de      Gluckstr. 24/42655 Solingen/Germany             */
/* This program is free software; you can redistribute it and/or modify it  */
/* under the terms of the GNU General Public License as published by the    */
/* Free Software Foundation; either version 2 of the License, or (at your   */
/* option) any later version.                                               */
/* This Program is distributed in the hope that it will be useful, but      */
/* WITHOUT ANY WARRANTY; without even the implied warranty of               */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPUOSE. See the GNU General*/
/* Public License for more details.                                         */
/* You should have received a copy of the GNU General Publiv License along  */
/* with this program; if not, write to the Free Software Foundation, Inc.,  */
/* 675 Mass Ave, Cambridge, MA 02139, USA.                                  */
/****************************************************************************/

#ifndef TVECTOR_H

#define TVECTOR_H

#include<math.h>

class TVector
{
public:
  double x,y,z;

  TVector() {};

  TVector(const double px, const double py, const double pz)
    {x=px; y=py; z=pz;};

  void init(const double px, const double py, const double pz)
    {x=px; y=py; z=pz;};

  double operator[](int i) const
  {
    return ((double *)this)[i];
  }

  TVector(TVector const &v1, TVector const &v2)            // Kreuzprodukt
    {x=v1.y*v2.z-v1.z*v2.y;
     y=v1.z*v2.x-v1.x*v2.z;
     z=v1.x*v2.y-v1.y*v2.x;};

  void init(TVector const &v1, TVector const&v2)
    {x=v1.y*v2.z-v1.z*v2.y;
     y=v1.z*v2.x-v1.x*v2.z;
     z=v1.x*v2.y-v1.y*v2.x;};


  TVector operator +=(TVector const &v2)
    {x+=v2.x; y+=v2.y; z+=v2.z; return *this;};
  TVector operator -=(TVector const &v2)
    {x-=v2.x; y-=v2.y; z-=v2.z; return *this;};
  TVector operator *=(double const f)
    {x*=f;y*=f;z*=f; return *this;}
  TVector operator /=(double const f)
    {x/=f;y/=f;z/=f; return *this;}

  TVector operator +(TVector const &v2) const
    {TVector ret=*this; ret+=v2; return ret;};
  TVector operator -(TVector const &v2) const
    {TVector ret=*this; ret-=v2; return ret;};
  TVector operator *(double f) const
    {TVector ret=*this; ret*=f; return ret;};
  TVector operator /(double f) const
    {TVector ret=*this; ret/=f; return ret;};
  TVector operator -() const
    {TVector ret; ret.x=-x; ret.y=-y; ret.z=-z;return ret;};

  double operator*(TVector const &v2) const
    {return x*v2.x+y*v2.y+z*v2.z;};
  operator double() const
    {return sqrt(x*x+y*y+z*z);};
  double len2() const
    { return x*x+y*y+z*z;};

  char operator ==(TVector const &v2) const {return x==v2.x && y==v2.y && z==v2.z;};
  
  double cosangle(TVector const &v2) const
  {
    double l=((double) *this) * (double)v2;
    return (l>0)?(*this * v2)/l : 2;
  }

  void normalize()
    {*this/=(double)*this;};

  TVector spiegel(TVector const &v2) const
  {
    TVector ret, l2;
    double tmp;
    tmp=*this * v2;
    tmp/=v2*v2;
    l2=v2*tmp;
    ret=-*this+l2+l2;
    return ret;
  }

  TVector spiegelnorm(TVector const &v2) const
  {
    TVector ret, l2;
    double tmp;
    tmp=*this * v2;
//    tmp/=v2*v2;
    l2=v2*tmp;
    ret=-*this+l2+l2;
    return ret;
  }
    
  void rotate(double ax, double ay, double az)
  {
    double sina,cosa;
    double ox,oy,oz;

    if (ax!=0.0)
    {
      sina=sin(ax);//sintab[angle2tab(ax)];
      cosa=cos(ax);//sintab[angle2tabcos(ax)];
      oy=y;oz=z;
      y=oy*cosa-oz*sina;
      z=oy*sina+oz*cosa;
    }
    if (ay!=0.0)
    {
      sina=sin(ay);//sintab[angle2tab(ay)];
      cosa=cos(ay);//sintab[angle2tabcos(ay)];
      ox=x;oz=z;
      x=ox*cosa+oz*sina;
      z=-ox*sina+oz*cosa;
    }
    if (az!=0.0)
    {
      sina=sin(az);//sintab[angle2tab(az)];
      cosa=cos(az);//sintab[angle2tabcos(az)];
      ox=x;oy=y;
      x=ox*cosa-oy*sina;
      y=ox*sina+oy*cosa;
    }
  };

/* Quadrat des Abstandes Punkt-Gerade */
  double abstpg2(TVector const &st, TVector const &dir) const
  {
    return TVector(*this-st, dir).len2();
  }

  TVector brech(TVector const &norm, double n) const
  {
    double cosi,cost,determ;
    TVector mynorm=norm;
    TVector brech;

    cosi=-cosangle(norm);
    if (cosi<0)
    {
      cosi=-cosi;
      mynorm=-mynorm;
    }
    else
      n=1.0/n;
    determ=1+n*n*(cosi*cosi-1);
    if (determ<0)
    {
      return spiegel(norm);
    }
    cost=sqrt(determ);
    brech=(*this)*n;
    mynorm*=n*cosi-cost;
    brech+=mynorm;
    return brech;
  }

};

#endif
